import React, {useCallback, useEffect, useRef} from 'react';
import {useSelector} from 'react-redux';
import {useHistory} from 'react-router';
import Cookie from 'js-cookie';

import environmentConfigSelector from 'selectors/common/environmentConfigSelector';

import {useLeaving} from 'utilities/hooks/userBehavior/useLeaving';
import {useNeedToLoadDynamicResource} from 'utilities/hooks/useNeedToLoadDynamicResource';
import {isE2ETesting} from 'utilities/testUtils/isE2ETesting';

import './UxfeedbackIFrame.scss';
import {usePlatform} from 'contexts/PlatformContext';

const UxfeedbackIFrame: React.FC = () => {
    const canRenderFrame = useNeedToLoadDynamicResource({
        timeoutProportion: 10,
        maxProportionTimeout: 3000,
    });
    const iframeRef = useRef<HTMLIFrameElement | null>(null);
    const oldLocation = useRef<Location>();
    const {uxfeedbackIframeSrc} = useSelector(environmentConfigSelector);
    const history = useHistory();
    const isLeaving = useLeaving();
    const isE2E = isE2ETesting();
    const {isWeb} = usePlatform();

    const sendMessage = useCallback(
        (type: string, payload?: object | string | number) => {
            iframeRef.current?.contentWindow?.postMessage(
                {type: `uxfb/${type}`, payload},
                '*',
            );
        },
        [],
    );

    const reachYaMetricaGoal = useCallback((id, params) => {
        const ym = window.ym;
        // @ts-ignore
        const yc = window['yaCounter' + id];

        if (typeof ym !== 'undefined') {
            ym(id, 'reachGoal', 'UX_FEEDBACK', params);
        } else if (yc) {
            yc.reachGoal('UX_FEEDBACK', params);
        }
    }, []);

    const handleScroll = useCallback(() => {
        sendMessage('scroll', {
            scrollTop: document.documentElement.scrollTop,
        });
    }, [sendMessage]);

    useEffect(() => {
        if (isLeaving) {
            sendMessage('event', {event: 'exitPage'});
        }
    }, [sendMessage, isLeaving]);

    const handleHistoryChange = useCallback(() => {
        sendMessage('history', {
            location: JSON.stringify(location),
            oldLocation: JSON.stringify({...oldLocation.current}),
        });
        oldLocation.current = {...location};
    }, [sendMessage]);

    const initLocationListener = useCallback(() => {
        history.listen(handleHistoryChange);
    }, [handleHistoryChange, history]);

    const getYaIdFromCookies = useCallback(() => {
        return Cookie.get('_ym_uid');
    }, []);

    const sendCurrentWinInfo = useCallback(() => {
        const clientHeight =
            window.innerHeight || document.documentElement.clientHeight;
        const ymId = getYaIdFromCookies();

        sendMessage('info', {
            url: document.location.href,
            cookies: document.cookie,
            ymId: ymId,
            devicePixelRatio: window.devicePixelRatio,
            scrollTop: document.documentElement.scrollTop,
            scrollHeight: document.documentElement.scrollHeight,
        });
        sendMessage('clientHeight', clientHeight);

        window.addEventListener(
            'resize',
            () => {
                const newClientHeight =
                    window.innerHeight || document.documentElement.clientHeight;

                sendMessage('clientHeight', newClientHeight);
            },
            false,
        );
    }, [sendMessage, getYaIdFromCookies]);

    const setIframePosition = useCallback(height => {
        const clientHeight =
            window.innerHeight || document.documentElement.clientHeight;
        const parsedHeight = parseInt(height, 10);
        const middle = (clientHeight - parsedHeight) / 2;
        const top = parsedHeight >= clientHeight ? 0 : middle;

        if (iframeRef.current) {
            iframeRef.current.style.transform = '';
            iframeRef.current.style.left = '';
            iframeRef.current.style.right = '';
            iframeRef.current.style.top = top + 'px';
        }
    }, []);

    const setIframeSize = useCallback((width, height) => {
        if (iframeRef.current) {
            if (width) {
                iframeRef.current.style.width = width;

                if (width === '100%') {
                    iframeRef.current.style.left = '0';
                    iframeRef.current.style.right = '0';
                    iframeRef.current.style.top = '0';
                    iframeRef.current.style.transform = '';
                }
            }

            if (height) {
                iframeRef.current.style.height = height;
            }
        }
    }, []);

    const setIframeStyle = useCallback(style => {
        if (iframeRef.current) {
            iframeRef.current.setAttribute('style', style);
            iframeRef.current.style.zIndex = '50000';
        }
    }, []);

    const showWidgetIcon = useCallback(
        (width, height) => {
            if (iframeRef.current) {
                iframeRef.current.setAttribute('width', width);
                iframeRef.current.setAttribute('height', height);
                iframeRef.current.className = '';
                iframeRef.current.style.opacity = '1';
            }

            sendMessage('showWidgetIcon', {});
        },
        [sendMessage],
    );

    const setIframeOpacity = useCallback(opacity => {
        if (iframeRef.current) {
            iframeRef.current.style.opacity = opacity;
        }
    }, []);

    const addClassToIframe = useCallback(className => {
        if (iframeRef.current) {
            iframeRef.current.classList.add(className);
        }
    }, []);

    const handleWindowMessage = useCallback(
        (e: MessageEvent) => {
            const {type = '', payload} = e.data;

            const [prefix, name] = type.split('/');

            if (prefix !== 'uxfb') {
                return;
            }

            if (name === 'style') {
                setIframeStyle(payload);
            }

            if (name === 'image') {
                showWidgetIcon(payload.width, payload.height);
            }

            if (name === 'opacity') {
                setIframeOpacity(payload);
            }

            if (name === 'class') {
                addClassToIframe('uxs-slide-back');
                addClassToIframe(payload);
            }

            if (name === 'size') {
                setIframeSize(payload.width, payload.height);
            }

            if (name === 'position') {
                setIframePosition(payload.height);
            }

            if (name === 'listener') {
                sendCurrentWinInfo();
            }

            if (name === 'locationListener') {
                initLocationListener();
            }

            if (name === 'addScroll') {
                window.addEventListener('scroll', handleScroll);
            }

            if (name === 'removeScroll') {
                window.removeEventListener('scroll', handleScroll);
            }

            if (name === 'reachYaMetricaGoal') {
                const {id, params} = payload;

                reachYaMetricaGoal(id, params);
            }
        },
        [
            addClassToIframe,
            handleScroll,
            initLocationListener,
            sendCurrentWinInfo,
            reachYaMetricaGoal,
            setIframeSize,
            setIframePosition,
            setIframeOpacity,
            showWidgetIcon,
            setIframeStyle,
        ],
    );

    useEffect(() => {
        window.addEventListener('message', handleWindowMessage);

        oldLocation.current = {...location};

        return (): void => {
            window.removeEventListener('message', handleWindowMessage);
        };
    }, [handleWindowMessage, history]);

    return canRenderFrame && isWeb && !isE2E ? (
        <iframe
            frameBorder="0"
            id="uxfb_iframe"
            ref={iframeRef}
            src={uxfeedbackIframeSrc}
            sandbox="allow-scripts allow-same-origin allow-forms allow-popups"
            style={{position: 'fixed', display: 'none'}}
        />
    ) : null;
};

export default React.memo(UxfeedbackIFrame);
