import {delay} from 'utilities/async/delay';
import {logWarning} from 'utilities/logger/logWarning';
import {params} from 'utilities/metrika';

function isCorrectFormat(): {
    isCorrect: boolean;
    formats: {
        moscow: string;
        paris: string;
    };
} {
    const dtfMoscow = new Intl.DateTimeFormat('en-US', {
        hour12: false,
        timeZone: 'Europe/Moscow',
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
    });

    const dtfParis = new Intl.DateTimeFormat('en-US', {
        hour12: false,
        timeZone: 'Europe/Paris',
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
    });

    const moscowFormat = dtfMoscow.format(
        new Date('2021-09-22T13:09:22+05:00'),
    );
    const parisFormat = dtfParis.format(new Date('2021-09-22T13:09:22+05:00'));

    return {
        isCorrect:
            moscowFormat === '09/22/2021, 11:09:22' &&
            parisFormat === '09/22/2021, 10:09:22',
        formats: {
            moscow: moscowFormat,
            paris: parisFormat,
        },
    };
}

export default async function checkIntlSupport(): Promise<void> {
    await delay(10000);

    let tryToLoadPolyfill = false;
    let supportState = 'fullSupport';

    try {
        if (!Intl) {
            supportState = 'noIntl';
        } else if (!Intl.DateTimeFormat) {
            supportState = 'noIntl.DateTimeFormat';
        } else if (!Intl.DateTimeFormat.prototype.formatToParts) {
            supportState = 'noIntl.DateTimeFormat.formatToParts';
        } else {
            const {isCorrect, formats} = isCorrectFormat();

            if (!isCorrect) {
                logWarning({
                    message: '[YATRAVEL] Intl. Неверное форматирование',
                    additional: formats,
                });

                tryToLoadPolyfill = true;
            }
        }
    } catch (err) {
        logWarning(
            {
                message: '[YATRAVEL] Intl.',
            },
            err,
        );

        tryToLoadPolyfill = true;
    }

    if (tryToLoadPolyfill) {
        try {
            await import(
                // @ts-ignore
                'date-time-format-timezone/build/src/date-time-format-timezone-all-zones-no-locale'
            );

            const {isCorrect, formats} = isCorrectFormat();

            if (isCorrect) {
                supportState = 'fullSupportWithPolyfill';
            } else {
                logWarning({
                    message:
                        '[YATRAVEL] Intl. С поллифилом. Неверное форматирование',
                    additional: formats,
                });

                supportState = 'incorrectFormatWithPolyfill';
            }
        } catch (err) {
            logWarning(
                {
                    message: '[YATRAVEL] Intl. С поллифилом.',
                },
                err,
            );

            supportState = 'error';
        }
    }

    params({
        common: {
            intlSupport: supportState,
        },
    });
}
