import React, {useContext, useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';

import {
    IPlatformState,
    platformInitialState,
} from 'reducers/avia/platform/reducer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {logError} from 'utilities/logger/logError';
import IPrice from 'utilities/currency/PriceInterface';
import {IHotelsSearchParams} from 'projects/hotels/utilities/urls/getHotelsSearchUrl';

export interface IPlatformContext extends IPlatformState {}

export const PlatformContext =
    React.createContext<IPlatformContext>(platformInitialState);

export function usePlatform(): IPlatformContext {
    return useContext(PlatformContext);
}

export interface INotifierBuyTickets {
    offer: IResultAviaVariant;
    redirectUrl: string;
}

export interface INotifierAddToFavorites {
    offersCount: number;
    minTariff: Nullable<IPrice>;
}

class Notifier {
    private platform: IPlatformContext;

    private readonly notify: TNotifyPlatform;

    constructor(platform: IPlatformContext, notify: TNotifyPlatform) {
        this.platform = platform;
        this.notify = notify.bind(this);
    }

    [EAppActions.ADD_TO_FAVORITES](payload: INotifierAddToFavorites): void {
        this.notify(EAppActions.ADD_TO_FAVORITES, {
            offersCount: payload.offersCount,
            minTariff: payload.minTariff,
        });
    }

    [EAppActions.REMOVE_FROM_FAVORITES](): void {
        this.notify(EAppActions.REMOVE_FROM_FAVORITES);
    }

    [EAppActions.SHARE](payload: any): void {
        this.notify(EAppActions.SHARE, {
            url: payload.url,
            title: payload.title,
        });
    }

    [EAppActions.SHOW_SEARCH_FORM](): void {
        this.notify(EAppActions.SHOW_SEARCH_FORM);
    }

    [EAppActions.BUY_TICKETS](payload: INotifierBuyTickets): void {
        this.notify(EAppActions.BUY_TICKETS, {
            offer: payload.offer,
            redirectUrl: payload.redirectUrl,
        });
    }

    [EAppActions.OPEN_HISTORY](): void {
        if (this.isAndroid()) {
            this.notify(EAppActions.OPEN_HISTORY);
        }
    }

    [EAppActions.OPEN_MENU](): void {
        if (this.isAndroid()) {
            this.notify(EAppActions.OPEN_MENU);
        }
    }

    [EAppActions.REPORT_METRICS_EVENT](payload: {
        name: string;
        json?: object;
    }): void {
        if (this.isAndroid()) {
            // На странице редиректа у нас нет интерфейса из-за CustomTabs.
            try {
                this.notify(EAppActions.REPORT_METRICS_EVENT, payload);
            } catch (e) {
                // ignore error
            }
        }
    }

    [EAppActions.NAVIGATE_TO_INDEX_PAGE](): void {
        this.notify(EAppActions.NAVIGATE_TO_INDEX_PAGE);
    }

    [EAppActions.NAVIGATE_TO_SEARCH_HOTELS_PAGE](): void {
        this.notify(EAppActions.NAVIGATE_TO_SEARCH_HOTELS_PAGE);
    }

    [EAppActions.NAVIGATE_TO_TRIPS_PAGE](): void {
        this.notify(EAppActions.NAVIGATE_TO_TRIPS_PAGE);
    }

    [EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE](): void {
        this.notify(EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE);
    }

    [EAppActions.NAVIGATE_TO_VARIANT_AVIA_PAGE](): void {
        this.notify(EAppActions.NAVIGATE_TO_VARIANT_AVIA_PAGE);
    }

    [EAppActions.NAVIGATE_TO_SEARCH_RESULT_HOTELS_PAGE](payload: {
        region?: IGeoRegion;
        searchParams: IHotelsSearchParams;
    }): void {
        this.notify(EAppActions.NAVIGATE_TO_SEARCH_RESULT_HOTELS_PAGE, payload);
    }

    [EAppActions.NAVIGATE_TO_VARIANT_HOTELS_PAGE](payload: {
        hotelSlug?: number;
        searchParams: IHotelsSearchParams;
        url: string;
    }): void {
        this.notify(EAppActions.NAVIGATE_TO_VARIANT_HOTELS_PAGE, payload);
    }

    [EAppActions.NAVIGATE_TO_EXTERNAL_URL](url: string): void {
        this.notify(EAppActions.NAVIGATE_TO_EXTERNAL_URL, {url});
    }

    [EAppActions.CREATE_HOTEL_ORDER](orderId: string): void {
        this.notify(EAppActions.CREATE_HOTEL_ORDER, {orderId});
    }

    [EAppActions.CREATE_AVIA_ORDER](orderId: string): void {
        this.notify(EAppActions.CREATE_AVIA_ORDER, {orderId});
    }

    [EAppActions.MOUNT_TRUST_STUB](purchaseToken: string): void {
        this.notify(EAppActions.MOUNT_TRUST_STUB, {purchaseToken});
    }

    [EAppActions.SET_AVIA_TEST_CONTEXT](tokens: {
        variantTestContext: string;
        paymentTestContext: string;
    }): void {
        this.notify(EAppActions.SET_AVIA_TEST_CONTEXT, tokens);
    }

    [EAppActions.SET_PAYMENT_TEST_CONTEXT](
        paymentTestContextToken: string,
    ): void {
        this.notify(EAppActions.SET_PAYMENT_TEST_CONTEXT, {
            paymentTestContextToken,
        });
    }

    [EAppActions.NAVIGATE_TO_FAVORITES](): void {
        this.notify(EAppActions.NAVIGATE_TO_FAVORITES);
    }

    [EAppActions.OPEN_AUTHORIZATION](): void {
        this.notify(EAppActions.OPEN_AUTHORIZATION);
    }

    protected logInvocationError(message: string, e: Error | unknown): void {
        logError(
            {
                message: `[YATRAVEL][AVIA] ${message}`,
                block: 'coordinator',
            },
            e,
        );
    }

    // эта проверка здесь из-за громоздких typeguard'ов, в идеале, конечно,
    // хотелось бы иметь отдельные классы для разных платформ
    protected isAndroid = (): boolean => this.platform.isAndroid;

    protected isFakeApp = (): boolean => this.platform.isFakeApp;
}

type TNotifyPlatform = <T>(action: EAppActions, payload?: T) => void;

function notifyIos<T>(this: Notifier, action: EAppActions, payload?: T): void {
    if (!window.webkit || !window.webkit.messageHandlers) {
        if (!this.isFakeApp()) {
            throw new ReferenceError('iOS messaging transport is missing');
        }

        window.webkit = {
            messageHandlers: {
                notification: {
                    postMessage: ({
                        action: a,
                        payload: p,
                    }: {
                        action: EAppActions;
                        payload?: T;
                    }): void => {
                        console.log(a, p);
                    },
                },
            },
        };
    }

    window.webkit.messageHandlers.notification.postMessage({
        action,
        payload,
    });
}

function notifyAndroid<T>(
    this: Notifier,
    action: EAppActions,
    payload?: T,
): void {
    if (!window.AndroidNativeInterface) {
        if (!this.isFakeApp()) {
            // Не падаем под андроидом, так как хром может прийти с куками приложения (привет CustomTabs)
            return;
        }

        window.AndroidNativeInterface = {};
    }

    if (!window.AndroidNativeInterface[action]) {
        if (!this.isFakeApp()) {
            throw new ReferenceError(
                '"' + action + '" method is not implemented',
            );
        }

        window.AndroidNativeInterface[action] = function (data: T): void {
            console.log(action, data);
        };
    }

    try {
        window.AndroidNativeInterface[action](JSON.stringify(payload || {}));
    } catch (e) {
        this.logInvocationError(action, e);
    }
}

function createNotifier(platform: IPlatformContext): Notifier {
    if (platform.isIos) {
        return new Notifier(platform, notifyIos);
    }

    if (platform.isAndroid) {
        return new Notifier(platform, notifyAndroid);
    }

    return {} as Notifier;
}

export interface ICoordinator {
    doAction: <T>(
        name: EAppActions,
        payload?: T | undefined,
    ) => true | undefined;
    canProcess: (name: EAppActions) => boolean;
}

export function createCoordinator(platform: IPlatformContext): ICoordinator {
    const notifier = createNotifier(platform);

    const canProcess = (name: EAppActions): boolean => Boolean(notifier[name]);

    const doAction = <T>(name: EAppActions, payload?: T): true | undefined => {
        if (canProcess(name)) {
            notifier[name](payload);

            return true;
        }
    };

    return {
        doAction,
        canProcess,
    };
}

export function useCoordinator(): ICoordinator {
    const platform = usePlatform();

    return useMemo(() => {
        return createCoordinator(platform);
    }, [platform]);
}
