import React, {
    ReactNode,
    FunctionComponent,
    createContext,
    useState,
    useContext,
    useEffect,
    useRef,
} from 'react';

interface IRumUiEvent {
    type: string;
    start: number;
    page?: IRumSubpage;
}

export interface IRumUiContext {
    _events: IRumUiEvent[];
    measure(type: string): void;
    setSubpage(page: IRumSubpage): void;
}

const RumUiContext = createContext<IRumUiContext | null>(null);

export function useRumUi(): IRumUiContext {
    const ctx = useContext(RumUiContext);

    if (!ctx) {
        throw new Error('RumUiProvider was not found');
    }

    return ctx;
}

interface IRumUiConsumerProps {
    children: (rumUi: IRumUiContext) => React.ReactNode;
}

export function RumUiConsumer({children}: IRumUiConsumerProps): ReactNode {
    return <>{children(useRumUi())}</>;
}

/**
 * Measures delta time for event with specified **type**
 * and sends it to RUM
 *
 * @param type - rum ui event type to observe
 * @param deps - on which deps to react
 */
export function useRumObserveState(type: string, deps: unknown[]): void {
    const {_events} = useRumUi();
    const eventsRef = useRef(_events);

    eventsRef.current = _events;

    // useEffect is called after all dom mutation applied and painted
    useEffect(() => {
        const events = eventsRef.current;
        const eventIdx = events.findIndex(x => x.type === type);

        if (eventIdx !== -1) {
            const {start, type, page} = events.splice(eventIdx, 1)[0];
            const time = window.Ya.Rum.getTime();

            if (window.Ya.Rum.sendDelta) {
                window.Ya.Rum.sendDelta(type, time - start, page);
            }
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, deps);
}

interface IRumUiProviderProps {
    page?: IRumSubpage;
}

export const RumUiProvider: FunctionComponent<IRumUiProviderProps> = ({
    children,
    page,
}) => {
    const [contextValue] = useState(() => createRumContext(page));

    return (
        <RumUiContext.Provider value={contextValue}>
            {children}
        </RumUiContext.Provider>
    );
};

interface IRumUiSubpageProps {
    page: string;
}

export function withRumUiSubpage(page: string) {
    return function <TProps>(
        Component: React.ComponentType<TProps>,
    ): React.ComponentType<TProps> {
        function Wrapped(props: TProps): JSX.Element {
            return (
                <RumUiSubpage page={page}>
                    <Component {...props} />
                </RumUiSubpage>
            );
        }

        Wrapped.displayName = `withRumUiSubpage(${Component.displayName})`;

        return Wrapped;
    };
}

export const RumUiSubpage: FunctionComponent<IRumUiSubpageProps> = ({
    children,
    page,
}) => {
    const rumUi = useRumUi();

    if (__CLIENT__ && window.Ya.Rum.makeSubPage) {
        rumUi.setSubpage(window.Ya.Rum.makeSubPage(page));
    }

    return <>{children}</>;
};

function createRumEvent(type: string, page?: IRumSubpage): IRumUiEvent {
    return {
        type,
        start: window.Ya.Rum.getTime(),
        page,
    };
}

function createRumContext(page?: IRumSubpage): IRumUiContext {
    if (__SERVER__) {
        return {} as any;
    }

    let curPage = page;

    const _events = Array.of<IRumUiEvent>();

    return {
        _events,
        measure(type: string): void {
            _events.push(createRumEvent(type, curPage));
        },
        setSubpage(newPage: IRumSubpage): void {
            curPage = newPage;
        },
    };
}
