import {useRef, useState} from 'react';

import {useEventListener} from 'utilities/hooks/useEventListener';

import {HideBodyVerticalScrollClassName} from 'components/HideBodyVerticalScroll/HideBodyVerticalScroll';

export enum EUseHeaderScrollSlideDirection {
    UP = 'up',
    DOWN = 'down',
}

export interface IUseHeaderScrollSlide {
    isRaised: boolean;
    isVisible: boolean;
    isFixed: boolean;
    direction: EUseHeaderScrollSlideDirection;
}

export interface IUseHeaderScrollSlideProps {
    hideThreshold?: number;
    riseThreshold?: number;
}

const HIDE_THRESHOLD = 70;
const RISE_THRESHOLD = 0;

export const useHeaderScrollSlide = (
    props: IUseHeaderScrollSlideProps = {},
): IUseHeaderScrollSlide => {
    const {hideThreshold = HIDE_THRESHOLD, riseThreshold = RISE_THRESHOLD} =
        props;
    const prevScrollTop = useRef<number>(0);
    const [isVisible, setVisibility] = useState(true);
    const [isRaised, setRaised] = useState(false);
    const [isFixed, setIsFixed] = useState(true);
    const [direction, setDirection] = useState(
        EUseHeaderScrollSlideDirection.DOWN,
    );

    const onScroll = (): void => {
        if (
            document.documentElement.classList.contains(
                HideBodyVerticalScrollClassName,
            )
        )
            return;

        const scrollY = window.scrollY;

        setIsFixed(scrollY <= hideThreshold && prevScrollTop.current > scrollY);

        const newDirection =
            prevScrollTop.current > scrollY
                ? EUseHeaderScrollSlideDirection.UP
                : EUseHeaderScrollSlideDirection.DOWN;

        setDirection(newDirection);

        const nextVisible =
            prevScrollTop.current > scrollY || scrollY < hideThreshold;
        const nextRaised = scrollY > riseThreshold;

        prevScrollTop.current = scrollY;

        if (isRaised !== nextRaised) {
            setRaised(nextRaised);
        }

        if (nextVisible !== isVisible) {
            setVisibility(nextVisible);
        }
    };

    useEventListener('scroll', onScroll);

    return {
        isRaised,
        isVisible,
        isFixed,
        direction,
    };
};
