import {useCallback, useMemo, useRef, useState} from 'react';
import {useSelector} from 'react-redux';
import isEqual from 'lodash/isEqual';
import debounce from 'lodash/debounce';

import {TGetCrossSaleHotelsResponse} from 'server/api/HotelsCrossSaleApi/types/TGetCrossSaleHotelsResponse';
import IGetCrossSaleHotelsRequestParamsV3 from 'server/api/HotelsCrossSaleApi/types/IGetCrossSaleHotelsRequestParamsV3';

import {useOffersEndpointSelector} from 'selectors/common/hotelsCrossSale/useOffersEndpointSelector';

import {useAsyncState} from 'utilities/hooks/useAsyncState';
import fetchHotelsCrossSale from 'hooks/useHotelsCrossSale/utilities/fetchHotelsCrossSale/fetchHotelsCrossSale';

interface IHotelsCrossSale {
    data: TGetCrossSaleHotelsResponse | undefined;
    isLoading: boolean;
    isError: boolean;
    isSuccess: boolean;
    fetch: (params: IGetCrossSaleHotelsRequestParamsV3) => Promise<void>;
    reset: () => void;
}

export default function useHotelsCrossSale(): IHotelsCrossSale {
    const {
        isLoading,
        isError,
        isSuccess,
        loading,
        success,
        error,
        reset: resetAsyncState,
    } = useAsyncState();

    const useOffersEndpoint = useSelector(useOffersEndpointSelector);

    const requestParams = useRef<
        IGetCrossSaleHotelsRequestParamsV3 | undefined
    >();
    const [data, setData] = useState<TGetCrossSaleHotelsResponse | undefined>();

    const fetch = useCallback(
        async (params: IGetCrossSaleHotelsRequestParamsV3) => {
            if (isEqual(requestParams.current, params)) {
                return;
            }

            requestParams.current = params;

            try {
                loading();

                const res = await fetchHotelsCrossSale(
                    params,
                    useOffersEndpoint,
                );

                if (res) {
                    setData(res);
                }

                success();
            } catch (e) {
                error();

                setData(undefined);
            }
        },
        [error, loading, success, useOffersEndpoint],
    );

    const fetchWithDebounce = useMemo(() => {
        return debounce(fetch, 0);
    }, [fetch]);

    const reset = useCallback(() => {
        resetAsyncState();
        setData(undefined);
        requestParams.current = undefined;
    }, [resetAsyncState]);

    return {
        data,
        isLoading,
        isError,
        isSuccess,
        fetch: fetchWithDebounce,
        reset,
    };
}
