import {FunctionComponent, useMemo, useState} from 'react';
import groupBy from 'lodash/groupBy';
import mapValues from 'lodash/mapValues';
import pickBy from 'lodash/pickBy';
import uniq from 'lodash/uniq';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';

import useIcons from './hooks/useIcons';

import cx from './IconsShowcase.scss';

const ICONS_RE = /^(?<size>[\d]+)\/(?<name>[^/\s]+)\.tsx?$/;

const config = {
    filter(filePath: string): boolean {
        return ICONS_RE.test(filePath);
    },
    getName(filePath: string): string {
        const matchResult = ICONS_RE.exec(filePath);

        if (!matchResult || !matchResult.groups) {
            throw new Error('Unexpected filePath');
        }

        return matchResult.groups.name;
    },
    getSize(filePath: string): string {
        const matchResult = ICONS_RE.exec(filePath);

        if (!matchResult || !matchResult.groups) {
            throw new Error('Unexpected filePath');
        }

        return matchResult.groups.size;
    },
};

const IconsShowcase: FunctionComponent = () => {
    const icons = useIcons(config);

    const sizes = useMemo(() => {
        return uniq(icons.map(icon => icon.size)).sort((a, b) => {
            return Number(a) - Number(b);
        });
    }, [icons]);

    const groupedByNameAndSize = useMemo(() => {
        const grouppedByName = groupBy(icons, icon => icon.name);

        return mapValues(grouppedByName, group =>
            groupBy(group, icon => icon.size),
        );
    }, [icons]);

    const [filter, setFilter] = useState('');

    const filteredAndGroupedByNameAndSize = useMemo(
        () =>
            pickBy(groupedByNameAndSize, (_, key) =>
                key.toLowerCase().includes(filter.toLowerCase()),
            ),
        [filter, groupedByNameAndSize],
    );

    return (
        <Box inset="4" style={{background: '#f8f8f8'}}>
            <div style={{maxWidth: 640, margin: 'auto'}}>
                <Flex
                    below="6"
                    justifyContent="space-between"
                    alignItems="baseline"
                >
                    <Heading level={1}>Список иконок</Heading>
                    <a href="https://www.figma.com/file/I8EBpOVkBHKTHVrDyWuaaIhI/Travel.Styles?node-id=9294%3A1052">
                        <Text size="xl" weight="bold" color="link">
                            Figma
                        </Text>
                    </a>
                </Flex>
                <Box below="4" style={{position: '-webkit-sticky', top: '0'}}>
                    <Input
                        value={filter}
                        placeholder="start typing"
                        onChange={(evt): void => setFilter(evt.target.value)}
                    />
                </Box>
                <table style={{borderSpacing: 0, width: '100%'}}>
                    <thead>
                        <tr className={cx('row-header')}>
                            <th>
                                <Box inset="2">Имя</Box>
                            </th>

                            {sizes.map(size => {
                                return (
                                    <th key={size}>
                                        <Box inset="2">{size}</Box>
                                    </th>
                                );
                            })}
                        </tr>
                    </thead>
                    <tbody>
                        {Object.entries(filteredAndGroupedByNameAndSize).map(
                            ([name, components]) => {
                                return (
                                    <tr className={cx('row')} key={name}>
                                        <td>
                                            <Box inset="2">{name}</Box>
                                        </td>
                                        {sizes.map(size => {
                                            const Icon =
                                                components?.[size]?.[0]
                                                    ?.Component;

                                            return (
                                                <td key={size}>
                                                    <Box inset="2">
                                                        {Icon && <Icon />}
                                                    </Box>
                                                </td>
                                            );
                                        })}
                                    </tr>
                                );
                            },
                        )}
                    </tbody>
                </table>
            </div>
        </Box>
    );
};

export default IconsShowcase;
