import {FunctionComponent, useMemo, useState} from 'react';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';
import useIcons, {IIcon} from 'icons/docs/hooks/useIcons';
import Flex from 'components/Flex/Flex';
import Modal from 'components/Modal/Modal';
import Text from 'components/Text/Text';

import cx from './IllustrationsShowcase.scss';

interface IIllustrationsShowcaseProps {
    title: string;
    figmaUrl: string;
    filter: (filePath: string) => boolean;
    getName: (filePath: string) => string;
    getSize: (filePath: string) => string;
}

const IllustrationsShowcase: FunctionComponent<IIllustrationsShowcaseProps> = ({
    title,
    figmaUrl,
    filter,
    getName,
    getSize,
}) => {
    const illustrations = useIcons({
        filter,
        getName,
        getSize,
    });

    const [activeIcon, setActiveIcon] = useState<IIcon | null>(null);

    const [inputFilter, setInputFilter] = useState('');
    const filteredIllustrations = useMemo(
        () =>
            illustrations.filter(illustration =>
                illustration.name
                    .toLowerCase()
                    .includes(inputFilter.toLowerCase()),
            ),
        [illustrations, inputFilter],
    );

    const ActiveIconComponent = activeIcon?.Component;

    return (
        <Box inset="4" style={{background: '#f8f8f8'}}>
            <div style={{maxWidth: 640, margin: 'auto'}}>
                <Flex
                    below="6"
                    justifyContent="space-between"
                    alignItems="baseline"
                >
                    <Heading level={1}>{title}</Heading>
                    <a href={figmaUrl}>
                        <Text size="xl" weight="bold" color="link">
                            Figma
                        </Text>
                    </a>
                </Flex>
                <Box below="4" style={{position: '-webkit-sticky', top: '0'}}>
                    <Input
                        value={inputFilter}
                        placeholder="start typing"
                        onChange={(evt): void =>
                            setInputFilter(evt.target.value)
                        }
                    />
                </Box>
                <Flex flexDirection="column">
                    {filteredIllustrations.map(icon => {
                        const {Component, name} = icon;

                        return (
                            <Flex
                                key={name}
                                className={cx('item')}
                                inline
                                inset="2"
                                between="8"
                                alignItems="center"
                                onClick={(): void => setActiveIcon(icon)}
                            >
                                <div className={cx('componentWrapper')}>
                                    <Component className={cx('component')} />
                                </div>
                                <Heading level={3}>{name}</Heading>
                            </Flex>
                        );
                    })}
                </Flex>
                <Modal
                    isVisible={Boolean(activeIcon)}
                    onClose={(): void => {
                        setActiveIcon(null);
                    }}
                >
                    <Modal.Content className={cx('modalContent')}>
                        <Heading className={cx('modalHeading')} level={2}>
                            {activeIcon?.name}
                        </Heading>
                        {ActiveIconComponent && (
                            <ActiveIconComponent style={{width: '100%'}} />
                        )}
                    </Modal.Content>
                </Modal>
            </div>
        </Box>
    );
};

export default IllustrationsShowcase;
