import React, {useEffect} from 'react';
import {Redirect, Route, Switch} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {cancelOrderRefundPoll} from 'reducers/account/orders/actions';
import passengersReducer from 'reducers/account/passengers/reducer';
import docsReducer from 'reducers/account/docs/reducer';
import docTypesReducer from 'reducers/account/docTypes/reducer';
import ordersReducer from 'reducers/account/orders/reducer';
import genericOrdersReducer from 'reducers/account/genericOrder/reducer';
import bookAndPayReducer from 'reducers/hotels/bookAndPayPage/bookAndPayReducer';
import bookingReducer from 'reducers/avia/booking/reducer';
import {resetTripsList} from 'reducers/trips/tripsList/actions';

import accountRootSaga from 'sagas/account/rootSaga';

import {loadable} from 'utilities/pageLoadable';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useIsAuth} from 'utilities/hooks/useIsAuth';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AccountMenu from 'projects/account/components/AccountMenu/AccountMenu';
import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import AppRoute from 'components/AppRoute/AppRoute';
import AccountPageMeta from 'projects/account/components/Meta/AccountPageMeta';
import SearchOrder from 'projects/account/pages/TripsPage/components/SearchOrder/SearchOrder';
import SearchToursOrder from 'projects/account/pages/TripsPage/components/SearchToursOrder/SearchToursOrder';

import Passenger from 'projects/account/pages/Passengers/AccountPassengerPage';
import OrderPage from 'projects/account/pages/Order/OrderPage';
import TripsPage from 'projects/account/pages/TripsPage/TripsPage';
import TripPage from 'projects/account/pages/TripPage/TripPage';

import cx from './UserAccountApp.scss';

const NotFoundApp = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

const UserAccountApp: React.FC = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const isAuth = useIsAuth();

    useEffect(
        () => (): void => {
            dispatch(cancelOrderRefundPoll());
            dispatch(resetTripsList());
        },
        [dispatch],
    );

    return (
        <>
            <Switch>
                <AppRoute
                    exact
                    path={[
                        URLs.accountTrips,
                        URLs.accountTripsSearch,
                        URLs.accountToursSearch,
                    ]}
                    component={TripsPage}
                    meta={AccountPageMeta}
                />

                {isAuth && (
                    <Route
                        exact
                        path={URLs.accountPassenger}
                        render={(props): React.ReactNode => (
                            <LayoutDefault
                                project={EProjectName.ACCOUNT}
                                footerType={EFooterProject.ACCOUNT}
                                showNavigation={deviceType.isDesktop}
                                bgColor={ELayoutBgColor.LIGHT}
                                headerBorderBottomType={
                                    deviceType.isDesktop
                                        ? EHeaderBorderBottomType.SHORT
                                        : EHeaderBorderBottomType.FULL
                                }
                            >
                                <AccountMenu className={cx('accountMenu')} />
                                <Passenger {...props} deviceType={deviceType} />
                            </LayoutDefault>
                        )}
                    />
                )}

                <Route exact path={URLs.accountTrip} component={TripPage} />

                <Route exact path={URLs.accountOrder} component={OrderPage} />

                <Route>
                    <Redirect to={URLs.accountTrips} />
                </Route>

                <Route component={NotFoundApp} />
            </Switch>
            <Switch>
                <AppRoute
                    exact
                    path={URLs.accountTripsSearch}
                    meta={AccountPageMeta}
                    component={SearchOrder}
                />
                <AppRoute
                    exact
                    path={URLs.accountToursSearch}
                    meta={AccountPageMeta}
                    component={SearchToursOrder}
                />
            </Switch>
        </>
    );
};

export default withReducers([
    [ELoadableReducer.ACCOUNT_PASSENGERS, passengersReducer],
    [ELoadableReducer.ACCOUNT_DOCUMENTS, docsReducer],
    [ELoadableReducer.ACCOUNT_DOC_TYPES, docTypesReducer],
    [ELoadableReducer.ACCOUNT_ORDERS, ordersReducer],
    [ELoadableReducer.ACCOUNT_GENERIC_ORDER, genericOrdersReducer],
    [ELoadableReducer.HOTELS_BOOK_AND_PAY, bookAndPayReducer],
    [ELoadableReducer.AVIA_BOOKING, bookingReducer],
])(withSaga(accountRootSaga)(UserAccountApp));
