import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Button from 'components/Button/Button';
import TicketCatL from 'icons/illustrations/TicketCat/TicketCatL';

import cx from './AccountEmpty.scss';

interface IAccountEmptyProps extends IWithQaAttributes {
    isMobile: boolean;
    image: {
        width: number;
        height: number;
    };
    title?: string;
    middleText?: string;
    smallText?: React.ReactNode;
    button?: {
        onClick: () => void;
        text: string;
        theme?: string;
        size?: string;
        width?: string;
        view?: string;
        tone?: string;
    };
}

export default class AccountEmpty extends PureComponent<IAccountEmptyProps> {
    static defaultProps = {
        image: {
            width: 156,
            height: 181,
        },
        button: {
            onClick: _noop,
        },
    };

    private renderImage(): React.ReactNode {
        const {
            image: {width, height},
        } = this.props;

        return (
            <div className={cx('image')}>
                <TicketCatL width={width} height={height} />
            </div>
        );
    }

    private renderTitle(): React.ReactNode {
        const {isMobile, title} = this.props;

        if (title) {
            return (
                <div
                    className={cx('title', {
                        title_mobile: isMobile,
                    })}
                >
                    {title}
                </div>
            );
        }

        return null;
    }

    private renderMiddleText(): React.ReactNode {
        const {middleText} = this.props;

        if (middleText) {
            return <div className={cx('middleText')}>{middleText}</div>;
        }

        return null;
    }

    private renderText(): React.ReactNode {
        const {smallText} = this.props;

        if (smallText) {
            return <div className={cx('info')}>{smallText}</div>;
        }

        return null;
    }

    private renderButton(): React.ReactNode {
        const {button, isMobile} = this.props;

        if (button && button.text) {
            return (
                <Button
                    theme="primary"
                    size={isMobile ? 'l' : 'm-inset'}
                    onClick={button.onClick}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'add',
                    })}
                >
                    {button.text}
                </Button>
            );
        }

        return null;
    }

    render(): React.ReactNode {
        const {isMobile} = this.props;

        return (
            <div className={cx('wrapper')} {...prepareQaAttributes(this.props)}>
                <div
                    className={cx('block', {
                        block_mobile: isMobile,
                    })}
                >
                    {this.renderImage()}
                    {this.renderTitle()}
                    {this.renderMiddleText()}
                    {this.renderText()}
                    {this.renderButton()}
                </div>
            </div>
        );
    }
}
