import React, {useMemo} from 'react';
import {matchPath, useLocation} from 'react-router-dom';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {internalUrl} from 'utilities/url';

import * as i18nBlock from 'i18n/account';

import Container from 'components/Layouts/Container/Container';
import Flex from 'components/Flex/Flex';
import Tabs from 'components/Tabs/Tabs';

import cx from './AccountMenu.scss';

interface IAccountMenuProps extends IWithClassName, IWithQaAttributes {
    supportPhone?: React.ReactNode;
}

interface ILink extends IWithQaAttributes {
    to: string;
    title: string;
    metrikaGoal?: TMetrikaGoal;
    exact: boolean;
}

const AccountMenu: React.FC<IAccountMenuProps> = props => {
    const {className, supportPhone} = props;

    const deviceType = useDeviceType();
    const location = useLocation();

    const links: ILink[] = useMemo(() => {
        return [
            {
                to: internalUrl(URLs.accountTrips),
                title: i18nBlock.menuDotTrips(),
                exact: false,
                ...prepareQaAttributes('tripsTab'),
            },
            {
                to: internalUrl(URLs.accountPassengers),
                title: i18nBlock.menuDotPassengers(),
                metrikaGoal: ECommonGoal.ACCOUNT_TAB_PASSENGERS,
                exact: true,
                ...prepareQaAttributes('passengersTab'),
            },
        ];
    }, []);

    return (
        <Container className={className} {...prepareQaAttributes(props)}>
            <Flex
                inline
                between={deviceType.isMobile ? 5 : 7}
                alignItems="baseline"
            >
                <Tabs
                    className={cx('tabs')}
                    border={deviceType.isMobile}
                    bold={deviceType.isDesktop}
                    tabs={links}
                    getTabId={(tab): string => tab.to}
                    renderTab={({tab, tabProps}): React.ReactNode => {
                        const {to, title, exact, metrikaGoal} = tab;

                        const isActive = Boolean(
                            matchPath(location.pathname, {
                                path: to,
                                exact,
                            }),
                        );

                        return (
                            <Tabs.Tab
                                key={to}
                                {...tabProps}
                                active={isActive}
                                to={to}
                                onClick={
                                    metrikaGoal &&
                                    ((): void => {
                                        reachGoal(metrikaGoal);
                                    })
                                }
                                {...prepareQaAttributes({
                                    current: tab,
                                    parent: props,
                                })}
                            >
                                {title}
                            </Tabs.Tab>
                        );
                    }}
                />

                {supportPhone && (
                    <div
                        className={cx('supportPhone')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'supportPhone',
                        })}
                    >
                        {supportPhone}
                    </div>
                )}
            </Flex>
        </Container>
    );
};

export default AccountMenu;
