import React from 'react';

import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Modal, {EModalAnimationType, IModalProps} from 'components/Modal/Modal';
import Heading from 'components/Heading/Heading';
import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';

import cx from './AccountModal.scss';

interface IAccountModalProps
    extends IWithQaAttributes,
        Pick<IModalProps, 'verticalAlign'> {
    title?: string;
    backButtonTitle?: string;
    children: React.ReactNode;
    isVisible: boolean;
    allowFullScreen?: boolean;
    contentClassName?: string;
    autoclosable?: boolean;

    /**
     * Соответствует ли открытому модалу отдельный урл.
     */
    isOwnUrlForModal?: boolean;
    onClose: () => void;
}

const AccountModal: React.FC<IAccountModalProps> = props => {
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const {
        contentClassName,
        title,
        backButtonTitle,
        children,
        isVisible,
        onClose,
        allowFullScreen,
        verticalAlign = isMobile ? 'top' : 'middle',
        isOwnUrlForModal = false,
        autoclosable = true,
        ...rest
    } = props;

    const hasCloseButton = !isMobile || (isMobile && !allowFullScreen);

    const content = (
        <Modal.Content
            className={contentClassName}
            between="5"
            above={isMobile && hasCloseButton ? 4 : undefined}
        >
            {Boolean(title) && <Heading level="2">{title}</Heading>}

            {children}
        </Modal.Content>
    );

    const fullScreen = allowFullScreen && isMobile;

    const modalProps = {
        isVisible,
        isMobile,
        preventBodyScroll: isVisible,
        animation: fullScreen
            ? EModalAnimationType.PUSH_RIGHT
            : EModalAnimationType.APPEAR,
        hasCloseButton,
        fullScreen,
        verticalAlign,
        canRenderBackButton: true,
        onClose,
        autoclosable,
        containerClassName: cx(
            'modalContainer',
            deviceMods('modalContainer', deviceType),
        ),
        ...rest,
    };

    if (isMobile && allowFullScreen) {
        return (
            <ModalWithBackButton
                {...modalProps}
                backButtonText={backButtonTitle}
                withHistorySupport={!isOwnUrlForModal}
            >
                {content}
            </ModalWithBackButton>
        );
    }

    return (
        <ModalWithHistoryBack {...modalProps}>{content}</ModalWithHistoryBack>
    );
};

export default AccountModal;
