import React, {useState, useEffect} from 'react';
import {withRouter, RouteComponentProps} from 'react-router-dom';

import {URLs} from 'constants/urls';

import {IAccountOrdersError} from 'reducers/account/orders/types/IAccountOrdersError';

import {internalUrl} from 'utilities/url';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reloadPage} from 'utilities/browser/reloadPage';

import * as i18nBlock from 'i18n/account-Order';

import Heading from 'components/Heading/Heading';
import Modal from 'components/Modal/Modal';
import Box from 'components/Box/Box';
import Button from 'components/Button/Button';

import cx from './ErrorModal.scss';

interface IErrorModalProps extends RouteComponentProps, IWithQaAttributes {
    isMobile: boolean;
    isError: boolean;
    error: IAccountOrdersError | null;
    clearError: () => void;
}

const ErrorModal: React.FC<IErrorModalProps> = props => {
    const {isMobile, isError, error, history, clearError} = props;
    const [currentError, setCurrentError] = useState(false);
    const [previousError, setPreviousError] = useState(false);
    const refreshPage = (): void => {
        clearError();
        reloadPage();
    };
    const goToMain = (): void => {
        clearError();
        history.push(internalUrl(URLs.index));
    };

    useEffect(() => {
        if (isError && !previousError) {
            setCurrentError(true);
        }

        setPreviousError(isError);
    }, [isError, previousError]);

    return currentError ? (
        <Modal
            autoclosable={false}
            isVisible={currentError}
            isMobile={isMobile}
            hasCloseButton={false}
            className={cx('modifiedModal', 'wrapper')}
            {...prepareQaAttributes(props)}
        >
            <Modal.Content between="4">
                <Box className={cx('header')}>
                    <Heading level="2">
                        {i18nBlock.errorModalDotHeader()}
                    </Heading>
                </Box>
                {error && (
                    <Box
                        className={cx('text')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'text',
                        })}
                    >
                        {error.text}
                    </Box>
                )}
                <div className={cx('buttons')}>
                    <div className={cx('button')}>
                        <Button
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'goToMainButton',
                            })}
                            theme="primary"
                            size={isMobile ? 'l' : 'm-inset'}
                            width="max"
                            onClick={goToMain}
                        >
                            {i18nBlock.errorModalDotGoToMainButton()}
                        </Button>
                    </div>
                    <div className={cx('button')}>
                        <Button
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'retryButton',
                            })}
                            width="max"
                            size={isMobile ? 'l' : 'm-inset'}
                            onClick={refreshPage}
                        >
                            {i18nBlock.errorModalDotRetryButton()}
                        </Button>
                    </div>
                </div>
            </Modal.Content>
        </Modal>
    ) : null;
};

export default withRouter(ErrorModal);
