import {
    FormEventHandler,
    PureComponent,
    FocusEventHandler,
    ReactNode,
    ComponentType,
} from 'react';
import {ValidationErrors} from 'final-form';

import {IOS_OS_NAME} from 'constants/common';

import {IWithDeviceType} from 'types/withDeviceType';

import FormError from '../FormError/FormError';

type TFieldNode = HTMLInputElement | null;
type TFieldName = string;

type TSetRefFunc = (name: TFieldName) => (node: TFieldNode) => void;

interface IWrappedComponentDisplayProps extends IWithDeviceType {
    formErrors: ValidationErrors;
    canShowErrors: boolean;
    modalRef: HTMLElement | undefined | null;
    handleSubmit: FormEventHandler;
    finalFormManager?: boolean; // Признак того, что final-form является стэйт-мэнеджером формы
}

interface IFormComponentProps {
    onTrySubmit: () => void;
    handleSubmit: FormEventHandler;

    onFocusField: FocusEventHandler<HTMLElement>;
    onBlurField: () => void;
    focusFieldByName?: (name: TFieldName) => void;
    getFieldRefByName: (name: TFieldName) => TFieldNode;
    setFieldRefByName: TSetRefFunc;
}

interface IWithFormErrorsDisplayProps
    extends IFormComponentProps,
        IWrappedComponentDisplayProps {}

interface IFormErrorsDisplayState {
    submitting: boolean;
    currentFocusField: TFieldName | null;
}

interface IFieldRef {
    fieldName: string;
    fieldNode: HTMLInputElement;
}

interface IInvalidFieldText {
    errorFieldNode?: TFieldNode;
    errorTooltipText?: string;
}

function withFormErrorsDisplay<
    P extends IWithFormErrorsDisplayProps = IWithFormErrorsDisplayProps,
>(FormComponent: ComponentType<P>) {
    return class FormErrorsDisplay extends PureComponent<
        Omit<P, keyof IFormComponentProps> & Pick<P, 'handleSubmit'>,
        IFormErrorsDisplayState
    > {
        /* Instance props */

        private _fields: IFieldRef[] = [];

        state = {
            currentFocusField: null,
            submitting: false,
        };

        static defaultProps = {
            formErrors: {},
            canShowErrors: false,
            finalFormManager: false,
        };

        componentWillUnmount(): void {
            this.removeAllListeners();
        }

        /* Listeners */

        private removeAllListeners = (): void => {
            const allFields = this.getAllFields();

            allFields.forEach(({fieldNode}) =>
                this.removeFieldListeners(fieldNode),
            );
        };

        private removeFieldListeners = (fieldNode: TFieldNode): void => {
            if (fieldNode) {
                fieldNode.removeEventListener(
                    'focus',
                    this.handleFocusListener,
                );
                fieldNode.removeEventListener('blur', this.handleBlurListener);
            }
        };

        private addFieldListeners = (fieldNode: TFieldNode): void => {
            if (fieldNode) {
                fieldNode.addEventListener('focus', this.handleFocusListener);
                fieldNode.addEventListener('blur', this.handleBlurListener);
            }
        };

        /* Handlers */

        private handleTrySubmit = (): void => {
            // setTimeout(0) нужен, чтобы props от `throw new SubmissionError(errors)` обновились к этому месту
            setTimeout(() => {
                this.setState(
                    {
                        submitting: true,
                    },
                    this.focusFirstErrorField,
                );
            });
        };

        private handleSubmit: FormEventHandler = (...args) => {
            const {handleSubmit, finalFormManager} = this.props;

            const resultOfSubmit = handleSubmit(...args);

            if (finalFormManager) {
                this.handleTrySubmit();
            }

            return resultOfSubmit;
        };

        private handleFocusListener = (e: FocusEvent): void => {
            const currentFocusField = this.getFieldNameByNode(
                e.target as HTMLInputElement,
            );

            this.setState({currentFocusField});
        };

        private handleBlurListener = (): void => {
            this.setState({currentFocusField: null});
        };

        private focusFirstErrorField = (): void => {
            const {formErrors} = this.props;
            const allFields = this.getAllFields();

            allFields.some(({fieldName, fieldNode}) => {
                const isErrorField = formErrors[fieldName];

                if (isErrorField) {
                    this.focusFieldByNode(fieldNode);
                }

                return isErrorField;
            });

            this.setState({
                submitting: false,
            });
        };

        private focusFieldByNode = (fieldNode: TFieldNode): void => {
            const {
                deviceType: {os},
            } = this.props;

            if (fieldNode && typeof fieldNode.focus === 'function') {
                /* Хак для IOS для исправления бага:
                При повторном фокусе в то же поле (с действующим фокусом в нем) пропадает каретка в поле, но фокус есть

                Что делаю:
                создаем временный элемент input и фокусимся сначала в него
                А затем удаляем и возвращаем фокус обратно в нужное поле по таймауту */
                if (os.name === IOS_OS_NAME) {
                    const tempElement = document.createElement('input');

                    tempElement.style.position = 'absolute';
                    tempElement.style.top = '-999999px';
                    tempElement.style.left = '-999999px';
                    tempElement.style.height = '0';
                    tempElement.style.width = '0';
                    tempElement.style.opacity = '0';
                    document.body.appendChild(tempElement);
                    tempElement.focus();

                    // requestAnimationFrame работает не правильно, отправляет скролл сильно выше
                    setTimeout(() => {
                        fieldNode.focus();
                        document.body.removeChild(tempElement);
                    }, 0);
                } else {
                    fieldNode.focus();
                }
            }
        };

        private focusFieldByName = (fieldName: TFieldName): void => {
            if (!fieldName) {
                return;
            }

            const fieldNode = this.getFieldRefByName(fieldName);

            this.focusFieldByNode(fieldNode);
        };

        /* Refs Manipulation */

        /**
         * Get all fields nodes
         *
         * @return {NodeList}
         */

        private getAllFields = (): IFieldRef[] => {
            return this._fields;
        };

        /**
         * Full field nodes list and add node listeners
         *
         * @param {string} fieldName - redux form field name
         *
         *  @return {Function} - setRefNode function
         */

        private setFieldRefByName: TSetRefFunc =
            fieldName =>
            (fieldNode): void => {
                if (fieldName && fieldNode) {
                    const allFields = this.getAllFields();
                    const isUpdatedField = allFields.some(currentField => {
                        const {fieldName: currentFieldName} = currentField;

                        if (currentFieldName === fieldName) {
                            currentField.fieldNode = fieldNode;

                            return true;
                        }

                        return false;
                    });

                    if (!isUpdatedField) {
                        allFields.push({fieldName, fieldNode});
                    }

                    this.removeFieldListeners(fieldNode);
                    this.addFieldListeners(fieldNode);
                }
            };

        /**
         * Get field node by field name
         *
         * @param {string} fieldName - redux form field name
         *
         * @return {Node} - fieldNode
         */

        private getFieldRefByName = (fieldName: TFieldName): TFieldNode => {
            const allFields = this.getAllFields();
            const currentField = allFields.find(
                item => item.fieldName === fieldName,
            );

            if (currentField) {
                return currentField.fieldNode;
            }

            return null;
        };

        /**
         * Get field name by field node
         *
         * @param {Node} fieldNode - field node
         *
         * @return {string} - field name
         */

        private getFieldNameByNode = (fieldNode: TFieldNode): TFieldName => {
            const allFields = this.getAllFields();
            const currentField = allFields.find(
                item => item.fieldNode === fieldNode,
            );

            if (currentField) {
                return currentField.fieldName;
            }

            return '';
        };

        /**
         * Get first error form field and error tooltip params
         *
         *  @return {Object} - fieldNode and tooltipText
         */

        private getFirstErrorFieldAndTooltipText = (): IInvalidFieldText => {
            const {currentFocusField} = this.state;
            const {formErrors, canShowErrors} = this.props;
            const fieldError = currentFocusField
                ? formErrors[currentFocusField]
                : '';

            if (canShowErrors) {
                if (currentFocusField && fieldError) {
                    const errorTooltipText = fieldError;
                    const errorFieldNode =
                        this.getFieldRefByName(currentFocusField);

                    return {
                        errorFieldNode,
                        errorTooltipText,
                    };
                }
            }

            return {};
        };

        private renderErrorTooltip = (): ReactNode => {
            const {modalRef} = this.props;
            const {errorTooltipText, errorFieldNode} =
                this.getFirstErrorFieldAndTooltipText();

            return (
                errorFieldNode &&
                errorTooltipText &&
                modalRef && (
                    <FormError
                        isVisible={Boolean(errorFieldNode)}
                        inputRef={errorFieldNode}
                        modalRef={modalRef}
                        error={errorTooltipText}
                    />
                )
            );
        };

        render(): ReactNode {
            return (
                <>
                    {this.renderErrorTooltip()}

                    <FormComponent
                        {...(this.props as P)}
                        onTrySubmit={this.handleTrySubmit}
                        onFocusField={this.handleFocusListener}
                        onBlurField={this.handleBlurListener}
                        focusFieldByName={this.focusFieldByName}
                        getFieldRefByName={this.getFieldRefByName}
                        setFieldRefByName={this.setFieldRefByName}
                        handleSubmit={this.handleSubmit}
                    />
                </>
            );
        }
    };
}

export default withFormErrorsDisplay;
