import React, {useCallback, useState} from 'react';
import {renderToString} from 'react-dom/server';
import {IYMapsInstance} from 'react-yandex-maps';

import {ICoordinates} from 'types/common/ICoordinates';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {getMapsUrl} from 'projects/account/components/HotelMapModal/utilities/getMapsUrl';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-TripPage';

import Modal from 'components/Modal/Modal';
import YandexMaps, {YandexMapPlacemark} from 'components/YandexMaps/YandexMaps';
import Button from 'components/Button/Button';
import CloseIcon from 'icons/16/Close';
import Heading from 'components/Heading/Heading';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import LocationIcon from 'icons/36/Location';

import cx from './HotelMapModal.scss';

interface IHotelMapModalProps extends IWithQaAttributes {
    visible: boolean;
    title: string;
    coordinates: ICoordinates;
    onClose(): void;
}

const HotelMapModal: React.FC<IHotelMapModalProps> = props => {
    const {visible, title, coordinates, onClose} = props;

    const [mapInstance, setMapInstance] = useState<IYMapsInstance | null>(null);
    const deviceType = useDeviceType();

    const {isDesktop, isMobile} = deviceType;

    const renderCloseButton = useCallback((close: () => void) => {
        return (
            <Button
                className={cx('closeButton')}
                shape="circle"
                theme="raised"
                onClick={close}
            >
                <CloseIcon />
            </Button>
        );
    }, []);

    return (
        <Modal
            className={cx('root', deviceMods('root', deviceType))}
            containerClassName={cx('contentContainer')}
            isVisible={visible}
            fullScreen={isMobile}
            renderCloseButton={isMobile ? renderCloseButton : undefined}
            onClose={onClose}
            {...prepareQaAttributes(props)}
        >
            <Modal.Content className={cx('modalContent')}>
                {isDesktop && (
                    <Heading
                        className={cx('title')}
                        level={2}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'title',
                        })}
                    >
                        {title}
                    </Heading>
                )}

                <YandexMaps
                    className={cx('mapContainer')}
                    customControlsClassName={cx('controls')}
                    onLoadMap={setMapInstance}
                    defaultCenter={coordinates}
                    hasZoomControl
                    hasGeoLocationControl
                >
                    {mapInstance && (
                        <YandexMapPlacemark
                            options={{
                                iconLayout:
                                    mapInstance.templateLayoutFactory.createClass(
                                        renderToString(
                                            <LocationIcon
                                                className={cx('markerIcon')}
                                            />,
                                        ),
                                    ),
                            }}
                            geometry={[coordinates.lat, coordinates.lon]}
                        />
                    )}

                    <ButtonLink
                        className={cx('howToGoButton')}
                        theme="raised"
                        url={getMapsUrl(coordinates)}
                        target="_blank"
                        rel="noopener noreferrer"
                    >
                        {i18nBlock.howToGo()}
                    </ButtonLink>
                </YandexMaps>
            </Modal.Content>
        </Modal>
    );
};

export default React.memo(HotelMapModal);
