import React, {useCallback, useEffect, useRef, useState} from 'react';
import setFieldData from 'final-form-set-field-data';

import {EFormKey} from 'constants/form/EFormKey';

import {IWithDeviceType} from 'types/withDeviceType';
import {EFieldName, EGroupName, IOrderSearchFormValues} from './types';
import {IWithClassName} from 'types/withClassName';
import {EAuthorizationOrderType} from 'server/api/OrdersAPI/types/IOrderAuthorization';

import history from 'utilities/browserHistory/browserHistory';
import validationInfo from 'projects/account/components/OrdesSearchForm/utilities/validateForm/orderSearchFormValidate';
import {accountURLs} from 'projects/account/utilities/urls';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrdersSearch';

import Form from 'components/Form/Form';
import focusFirstInvalidField from 'components/Form/mutators/focusFirstInvalidField';
import setFormErrors from 'components/Form/mutators/setFormErrors';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import OrderSearchFormContent from './components/OrdersSearchFormContent/OrdersSearchFormContent';

import {ordersBrowserProvider} from 'serviceProvider/orders/ordersBrowserProvider';

import cx from './OrdersSearchForm.scss';

const focusOnSubmitErrors = createSubmitErrorDecorator<IOrderSearchFormValues>(
    EFormKey.ACCOUNT_ORDER_SEARCH,
);

const initialValues: IOrderSearchFormValues = {
    [EGroupName.ORDERS_SEARCH]: {
        [EFieldName.USER_PRETTY_ID_FIELD_NAME]: '',
        [EFieldName.USER_PHONE_OR_EMAIL_FIELD_NAME]: '',
    },
};

interface IOrdersSearchForm
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    isAuth: boolean;
    authorizePath?: string;
    title?: React.ReactNode;
}

const OrdersSearchForm: React.FC<IOrdersSearchForm> = props => {
    const {deviceType, authorizePath, isAuth, title, className} = props;
    const [isLoading, setLoading] = useState(false);
    const [canRedirectToTravel, setCanRedirectToTravel] = useState(false);
    const [notFoundError, setNotFoundError] = useState('');

    useEffect(() => {
        setCanRedirectToTravel(true);
    }, []);

    const handleFormSubmit = useCallback(
        async (values: IOrderSearchFormValues): Promise<void> => {
            setLoading(true);

            try {
                const prettyOrderId =
                    values[EGroupName.ORDERS_SEARCH]?.[
                        EFieldName.USER_PRETTY_ID_FIELD_NAME
                    ];
                const userPhoneOrEmail =
                    values[EGroupName.ORDERS_SEARCH]?.[
                        EFieldName.USER_PHONE_OR_EMAIL_FIELD_NAME
                    ];

                if (!prettyOrderId) {
                    throw new Error('Нет prettyOrderId');
                }

                const data =
                    await ordersBrowserProvider.createOrderAuthorization({
                        secret: userPhoneOrEmail || '',
                        yandex_order_id: prettyOrderId,
                    });

                if (data) {
                    switch (data.orderType) {
                        case EAuthorizationOrderType.TRAIN:
                        case EAuthorizationOrderType.GENERIC:
                            history?.push(accountURLs.getOrderUrl(data.id));

                            break;
                        case EAuthorizationOrderType.HOTEL_EXPEDIA:
                            history?.push(accountURLs.getOrderUrl(data.id));

                            break;
                        case EAuthorizationOrderType.AVIA_AEROFLOT:
                            history?.push(accountURLs.getAviaOrderUrl(data.id));

                            break;
                        default:
                            throw new Error();
                    }
                }
            } catch (e) {
                setLoading(false);
                setNotFoundError(i18nBlock.errorDotNotFound());
            }
        },
        [],
    );

    const refModal = useRef(null);

    return (
        <div
            className={cx('root', className)}
            ref={refModal}
            {...prepareQaAttributes(props)}
        >
            <Form<IOrderSearchFormValues>
                initialValues={initialValues}
                validationInfo={validationInfo}
                onSubmit={handleFormSubmit}
                decorators={[focusOnSubmitErrors]}
                mutators={{
                    focusFirstInvalidField,
                    setFieldData,
                    setFormErrors,
                }}
                render={({handleSubmit}): React.ReactNode => (
                    <OrderSearchFormContent
                        isAuth={isAuth}
                        authorizePath={authorizePath}
                        isLoading={isLoading}
                        notFoundError={notFoundError}
                        canRedirectToTravel={canRedirectToTravel}
                        handleSubmit={handleSubmit}
                        deviceType={deviceType}
                        title={title}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'content',
                        })}
                    />
                )}
            />
        </div>
    );
};

export default React.memo(OrdersSearchForm);
