import React, {PureComponent} from 'react';
import {FieldRenderProps} from 'react-final-form';

import {YANDEX_ROOT_PATH} from 'constants/common';
import {EFormKey} from 'constants/form/EFormKey';

import {IWithDeviceType} from 'types/withDeviceType';
import {EFieldName, EGroupName} from '../../types';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrdersSearch';

import Input from 'components/Input/Input';
import FormField from 'components/FormField/FormField';
import FieldGroup from 'components/Form/components/FieldGroup/FieldGroup';
import FullscreenSpinner from 'components/FullscreenSpinner/FullscreenSpinner';
import OrdersSearchFormAuthorizeLink from 'projects/account/components/OrdesSearchForm/components/OrdersSearchFormAuthorizeLink';
import OrderSearchFormSubmitButton from 'projects/account/components/OrdesSearchForm/components/OrderSearchFormSubmitButton/OrderSearchFormSubmitButton';

import cx from './OrdersSearchFormContent.scss';

/* Props */
interface IOrdersSearchFormContentProps
    extends IWithDeviceType,
        IWithQaAttributes {
    // Props from FormErrorsDisplay
    handleSubmit: () => void;
    authorizePath?: string;
    isAuth: boolean;
    isLoading: boolean;
    notFoundError: string;
    canRedirectToTravel: boolean;
    title?: React.ReactNode;
}

class OrdersSearchFormContent extends PureComponent<IOrdersSearchFormContentProps> {
    private onInputChange =
        (handleChange: FieldRenderProps<string>['input']['onChange']) =>
        (_: any, value: string): void =>
            handleChange(value);

    private renderTitle = (): React.ReactNode => (
        <div
            className={cx('title')}
            {...prepareQaAttributes({parent: this.props, current: 'title'})}
        >
            {i18nBlock.title()}
        </div>
    );

    private renderText(): React.ReactNode {
        const {authorizePath, isAuth, canRedirectToTravel} = this.props;

        if (!isAuth) {
            let authorizeLink = `${authorizePath}`;

            if (canRedirectToTravel) {
                authorizeLink += document.location.href;
            } else {
                authorizeLink += YANDEX_ROOT_PATH;
            }

            return (
                <div className={cx('text')}>
                    {insertJSXIntoKey(i18nBlock.small)({
                        link: (
                            <OrdersSearchFormAuthorizeLink
                                authorizeLink={authorizeLink}
                                className={cx('link')}
                                {...prepareQaAttributes({
                                    parent: this.props,
                                    current: 'enterAccountLink',
                                })}
                            />
                        ),
                    })}
                </div>
            );
        }

        return null;
    }

    private renderFields = (): React.ReactNode => (
        <FieldGroup groupId={EGroupName.ORDERS_SEARCH}>
            <div className={cx('fields')}>
                {[
                    EFieldName.USER_PRETTY_ID_FIELD_NAME,
                    EFieldName.USER_PHONE_OR_EMAIL_FIELD_NAME,
                ].map((fieldName: string) => this.renderInput(fieldName))}
            </div>
        </FieldGroup>
    );

    private renderInput = (fieldName: string): React.ReactNode => (
        <FormField
            key={fieldName}
            deviceType={this.props.deviceType}
            name={fieldName}
            className={cx('field')}
            title={
                fieldName === EFieldName.USER_PHONE_OR_EMAIL_FIELD_NAME
                    ? i18nBlock.userPhoneOrEmailDotLabel()
                    : i18nBlock.prettyOrderIdDotLabel()
            }
            hint={
                fieldName === EFieldName.USER_PHONE_OR_EMAIL_FIELD_NAME
                    ? i18nBlock.userPhoneOrEmailDotSmall()
                    : i18nBlock.prettyOrderIdDotSmall()
            }
            control={(
                {input},
                {inputRef, controlRef, error},
            ): React.ReactElement<HTMLInputElement> => {
                const {deviceType} = this.props;

                return (
                    <div data-form-field>
                        <Input
                            {...input}
                            onChange={this.onInputChange(input.onChange)}
                            id={input.name}
                            inputRef={(currentInputRef): void => {
                                inputRef.current = currentInputRef;
                            }}
                            innerRef={controlRef}
                            state={error ? 'error' : undefined}
                            value={input.value}
                            size={deviceType.isMobile ? 'l' : 'm'}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: `${fieldName}Input`,
                            })}
                        />
                    </div>
                );
            }}
        />
    );

    private renderError(error: string): React.ReactNode {
        return (
            <div
                className={cx('notFound')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'notFoundError',
                })}
            >
                {error}
            </div>
        );
    }

    render(): React.ReactNode {
        const {
            deviceType: {isMobile},
            handleSubmit,
            isLoading,
            notFoundError,
            title = this.renderTitle(),
        } = this.props;

        return (
            <div
                className={cx('formContainer')}
                {...prepareQaAttributes(this.props)}
            >
                <form
                    name={EFormKey.ACCOUNT_ORDER_SEARCH}
                    className={cx('wrap', {
                        wrap_mobile: isMobile,
                    })}
                    onSubmit={handleSubmit}
                    noValidate
                >
                    {title}
                    {this.renderFields()}
                    {isLoading && <FullscreenSpinner />}
                    <OrderSearchFormSubmitButton
                        className={cx('button')}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'searchButton',
                        })}
                    />
                    {notFoundError && this.renderError(notFoundError)}
                    {this.renderText()}
                </form>
            </div>
        );
    }
}

export default OrdersSearchFormContent;
