import moment from 'moment';
import React, {useMemo} from 'react';

import {accountTrips} from 'constants/urls/account';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    TBindToursApiResponse,
    EBindingCreationStatus,
} from 'server/api/ToursApi/types/TBindToursApiResponse';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {getNow} from 'utilities/dateUtils';
import getResultTextByStatus from 'projects/account/components/ToursOrderSearchForm/utilities/getResultTextByStatus';

import * as i18nBlock from 'i18n/account-ToursOrdersSearch';

import Plus5M from 'icons/illustrations/Plus5/Plus5M';
import CommonErrorM from 'icons/illustrations/CommonError/CommonErrorM';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import Flex from 'components/Flex/Flex';
import YandexMessengerLink from 'components/YandexMessenger/components/YandexMessengerLink/YandexMessengerLink';

import cx from './SubmitResult.scss';

export interface ISubmitResultProps extends IWithClassName, IWithDeviceType {
    data: TBindToursApiResponse;
    resetForm: () => void;
}

const SubmitResult: React.FC<ISubmitResultProps> = ({
    className,
    data,
    resetForm,
    deviceType,
}) => {
    const isSuccess = [
        EBindingCreationStatus.CREATED,
        EBindingCreationStatus.ALREADY_BOUND_TO_CURRENT_USER,
    ].includes(data.creationStatus);

    const support = useMemo(
        () => (
            <YandexMessengerLink
                triggerClassName={cx('yandexMessenger')}
                theme="normal"
                entrypoint="tripsPage"
                metrikaGoal={ETripsGoal.TRIPS_PAGE_CHAT_CLICK}
                label={i18nBlock.support()}
                withIcon={false}
            />
        ),
        [],
    );

    return (
        <Flex
            className={cx(className, 'root')}
            flexDirection="column"
            justifyContent="center"
            alignItems="center"
        >
            {isSuccess ? (
                <Plus5M width="174" height="120" />
            ) : (
                <CommonErrorM width="120" height="120" />
            )}
            <Heading className={cx('header')} level="2">
                {isSuccess
                    ? i18nBlock.successTitle({
                          isPast: moment(getNow()).isAfter(
                              moment(data.expectedTopupDate),
                          ),
                      })
                    : i18nBlock.somethingWentWrong()}
            </Heading>
            <Text className={cx('text')}>
                {getResultTextByStatus(data, support)}
            </Text>

            <div className={cx('buttonBlock')}>
                <ButtonLink
                    to={isSuccess ? accountTrips : undefined}
                    onClick={isSuccess ? undefined : resetForm}
                    size={deviceType.isMobile ? 'l' : 'm'}
                    theme="secondary"
                >
                    {isSuccess
                        ? i18nBlock.buttonSuccessText()
                        : i18nBlock.buttonTryAgainText()}
                </ButtonLink>
            </div>
        </Flex>
    );
};

export default SubmitResult;
