import React, {useCallback} from 'react';
import {Redirect, RouteComponentProps} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import {EAuthorizationOrderType} from 'server/api/OrdersAPI/types/IOrderAuthorization';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {clearError} from 'reducers/account/orders/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {orderDetailsSelector} from 'selectors/account/ordersSelector';

import {accountURLs} from 'projects/account/utilities/urls';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/account-Order';

import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import CheckOrderAuthorization from 'components/CheckOrderAuthorization/CheckOrderAuthorizationContainer';
import ErrorModal from 'projects/account/components/ErrorModal/ErrorModal';
import BookLoader from 'components/BookLoader/BookLoader';
import NotFoundContent from 'components/NotFound/NotFoundContent/NotFoundContent';
import OrderHotels from './components/OrderHotels/OrderHotels';
import GenericOrder from 'projects/account/pages/Order/components/GenericOrder/GenericOrder';

import OrderTrainsContainer from 'projects/account/pages/Order/containers/OrderTrainsContainer/OrderTrainsContainer';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './OrderPage.scss';

interface IOrderPageRouteParams {
    orderId: string;
}

interface IOrderPageProps extends RouteComponentProps<IOrderPageRouteParams> {}

const OrderPage: React.FC<IOrderPageProps> = props => {
    const {orderId} = props.match.params;

    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();

    const selector = useCallback(
        (state: StoreInterface) => orderDetailsSelector(state, {orderId}),
        [orderId],
    );
    const {orderType, needAuth, isNotFound, hasAccess, error, isError} =
        useSelector(selector);

    const dispatch = useDispatch();
    const handleClearError = useCallback(() => {
        dispatch(clearError());
    }, [dispatch]);

    const renderAuthorization = useCallback((): React.ReactNode => {
        return (
            <>
                <CheckOrderAuthorization
                    className={cx('orderAuth', {
                        orderAuth_hidden: !needAuth,
                    })}
                    orderId={orderId}
                />
                <BookLoader
                    title={i18nBlock.loaderDotTitle()}
                    description={i18nBlock.loaderDotDescription()}
                    isLoading={!needAuth && !isNotFound}
                />
            </>
        );
    }, [isNotFound, needAuth, orderId]);

    const renderOrder = useCallback((): React.ReactNode => {
        switch (orderType) {
            case EAuthorizationOrderType.TRAIN:
                return (
                    <OrderTrainsContainer
                        orderId={orderId}
                        orderType={EAuthorizationOrderType.TRAIN}
                    />
                );
            case EAuthorizationOrderType.GENERIC:
                return <GenericOrder id={orderId} />;
            case EAuthorizationOrderType.HOTEL_EXPEDIA:
                return <OrderHotels orderId={orderId} />;
            default:
                return <NotFoundContent />;
        }
    }, [orderId, orderType]);

    /**
     * Авиа заказ живет отдельно
     * сначала нужно дождаться hasAccess прежде чем делать редирект (так как в state может лежать старый orderType)
     */
    if (hasAccess && orderType === EAuthorizationOrderType.AVIA_AEROFLOT) {
        return <Redirect to={accountURLs.getAviaOrderUrl(orderId)} />;
    }

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            project={EProjectName.ACCOUNT}
            footerType={EFooterProject.ACCOUNT}
            showNavigation={deviceType.isDesktop}
            bgColor={
                deviceType.isMobile ? ELayoutBgColor.DARK : ELayoutBgColor.LIGHT
            }
            searchFormInitialIsExpanded={false}
            headerBorderBottomType={EHeaderBorderBottomType.NONE}
            footerIncludeSupportBlock
            footerClassName={cx('footer')}
            showHeader={isWeb}
            showFooter={isWeb}
        >
            {hasAccess ? renderOrder() : renderAuthorization()}
            <ErrorModal
                isMobile={deviceType.isMobile}
                error={error}
                isError={isError}
                clearError={handleClearError}
                {...prepareQaAttributes('errorModal')}
            />
        </LayoutDefault>
    );
};

export default React.memo(OrderPage);
