import React, {MouseEventHandler, useCallback, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/account-OrderTrains-Passenger';
import * as i18nHotelsFiscalReceiptBlock from 'i18n/hotels-FiscalReceipt';

import Link from 'components/Link/Link';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './DownloadReceiptLink.scss';

interface IRefundBlankLinkProps extends IWithClassName, IWithQaAttributes {
    receiptUrl: string | undefined;
    onClick?(): void;
}

const DownloadReceiptLink: React.FC<IRefundBlankLinkProps> = props => {
    const {className, receiptUrl, onClick} = props;
    const {
        value: popupIsVisible,
        setTrue: openPopup,
        setFalse: closePopup,
    } = useBoolean(false);

    const linkRef = useRef<HTMLElement>(null);

    const handleLinkClick: MouseEventHandler = useCallback(
        e => {
            if (!receiptUrl) {
                openPopup();

                e.preventDefault();

                return;
            }

            onClick?.();
        },
        [receiptUrl, openPopup, onClick],
    );

    return (
        <>
            <Link
                className={className}
                url={receiptUrl ?? ''}
                target="_blank"
                rel="noopener noreferrer"
                onClick={handleLinkClick}
                innerRef={linkRef}
                {...prepareQaAttributes(props)}
            >
                {i18nBlock.refundDotReceipt()}
            </Link>

            {!receiptUrl && (
                <MessageBoxPopup
                    anchorRef={linkRef}
                    isVisible={popupIsVisible}
                    onClose={closePopup}
                >
                    <div className={cx('popup')}>
                        {i18nHotelsFiscalReceiptBlock.emptyFiscalReceiptTitle()}
                    </div>
                </MessageBoxPopup>
            )}
        </>
    );
};

export default DownloadReceiptLink;
