import {FC, memo, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {batchActions} from 'redux-batched-actions';

import EServiceType from 'server/api/GenericOrderApi/types/common/EServiceType';
import {EAuthorizationOrderType} from 'server/api/OrdersAPI/types/IOrderAuthorization';
import EGenericOrderSource from 'server/api/GenericOrderApi/types/common/EGenericOrderSource';
import {EGenericOrderRefundStatus} from 'reducers/account/genericOrder/refund/status/types';

import requestGenericOrderItemAction from 'reducers/account/genericOrder/item/thunk/requestGenericOrderItemAction';
import {setRefundStatusAction} from 'reducers/account/genericOrder/refund/status/actions';
import {setIsRefundModalOpenedAction} from 'reducers/account/genericOrder/refund/isModalOpened/actions';

import genericOrderItemInfoSelector from 'selectors/account/genericOrder/item/genericOrderItemInfoSelector';
import refundStatusSelector from 'selectors/account/genericOrder/refund/refundStatusSelector';

import getFirstServiceType from 'projects/account/pages/Order/components/GenericOrder/utilities/getFirstServiceType';

import * as i18nBlock from 'i18n/account-Order';

import BookLoader from 'components/BookLoader/BookLoader';
import TrainHangingYandexMessenger from 'projects/trains/components/HangingYandexMessenger/HangingYandexMessenger';
import BusHangingYandexMessenger from 'projects/buses/components/HangingYandexMessenger/HangingYandexMessenger';
import BusesOrder from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/BusesOrder';

import OrderTrainsContainer from 'projects/account/pages/Order/containers/OrderTrainsContainer/OrderTrainsContainer';

interface IGenericOrderProps {
    id: string;
}

const GenericOrder: FC<IGenericOrderProps> = props => {
    const {id} = props;

    const dispatch = useDispatch();

    const orderInfo = useSelector(genericOrderItemInfoSelector);
    const refundStatus = useSelector(refundStatusSelector);

    const isRefunding = refundStatus !== EGenericOrderRefundStatus.NOT_STARTED;

    useEffect(() => {
        dispatch(
            requestGenericOrderItemAction({
                orderId: id,
                source: EGenericOrderSource.ORDER_PAGE,
            }),
        );

        return (): void => {
            dispatch(
                batchActions([
                    setRefundStatusAction(
                        EGenericOrderRefundStatus.NOT_STARTED,
                    ),
                    setIsRefundModalOpenedAction(false),
                ]),
            );
        };
    }, [dispatch, id]);

    if (!orderInfo.isFetched) {
        return (
            <BookLoader
                title={i18nBlock.loaderDotTitle()}
                description={i18nBlock.loaderDotDescription()}
                isLoading
            />
        );
    }

    if (orderInfo.isFailed) {
        return null;
    }

    const serviceType = getFirstServiceType(orderInfo.value.services);

    if (serviceType === EServiceType.TRAIN) {
        return (
            <>
                <OrderTrainsContainer
                    orderId={id}
                    orderType={EAuthorizationOrderType.GENERIC}
                />

                <TrainHangingYandexMessenger />
            </>
        );
    }

    if (serviceType === EServiceType.BUS) {
        return (
            <>
                <BusesOrder order={orderInfo.value} isRefunding={isRefunding} />

                <BusHangingYandexMessenger />
            </>
        );
    }

    return null;
};

export default memo(GenericOrder);
