import {FC, memo, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getBusService from 'projects/buses/utilities/booking/getBusService';

import * as i18nContacts from 'i18n/account-OrderContacts';

import OrderHeader from 'components/OrderHeader/OrderHeader';
import OrderInfo from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/OrderInfo/OrderInfo';
import OrderContacts from 'projects/account/pages/Order/components/OrderContacts/OrderContacts';
import Tickets from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/Passengers/Passengers';
import BusesRefundModal from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/BusesRefundModal/BusesRefundModal';
import OrderActions from 'projects/buses/components/OrderActions/OrderActions';
import {EOrderActionsSource} from 'components/OrderActions/OrderActions';
import OrderWrapper from 'projects/account/pages/Order/components/OrderWrapper/OrderWrapper';

import cx from './BusesOrder.scss';

interface IBusesOrderProps {
    order: IGenericOrderInfo;
    isRefunding: boolean;
}

const BusesOrder: FC<IBusesOrderProps> = props => {
    const {order, isRefunding} = props;

    const deviceType = useDeviceType();

    const service = useMemo(
        () => getBusService(order.services),
        [order.services],
    );

    const ride = service?.busInfo.ride;
    const tickets = service?.busInfo.tickets;
    const downloadBlankToken = service?.busInfo.downloadBlankToken;
    const refundPartInfo = service?.refundPartInfo;

    const ticketsWithPossibleRefundCount = useMemo(
        () =>
            tickets?.filter(
                ticket =>
                    ticket.refundPartInfo?.state === ERefundPartState.ENABLED,
            ).length ?? 0,
        [tickets],
    );

    const orderActions = useMemo(() => {
        if (!tickets || order.state === EGenericOrderState.REFUNDED) {
            return null;
        }

        return (
            <OrderActions
                className={cx('actions')}
                source={EOrderActionsSource.ORDER_PAGE}
                orderId={order.id}
                downloadBlankToken={downloadBlankToken}
                ticketsWithPossibleRefundCount={ticketsWithPossibleRefundCount}
                refundPartInfo={refundPartInfo || null}
                isRefunding={isRefunding}
            />
        );
    }, [
        isRefunding,
        order.id,
        order.state,
        refundPartInfo,
        tickets,
        ticketsWithPossibleRefundCount,
        downloadBlankToken,
    ]);

    return (
        <OrderWrapper className={cx('root', deviceMods('root', deviceType))}>
            <OrderHeader
                type={EProjectName.BUSES}
                number={order.prettyId}
                displayState={order.displayState}
                tripId={order.tripId}
            />

            {ride && (
                <OrderInfo
                    className={cx('info')}
                    ride={ride}
                    actions={deviceType.isMobile && orderActions}
                />
            )}

            {deviceType.isDesktop && orderActions}

            {tickets && (
                <Tickets
                    className={cx('tickets')}
                    tickets={tickets}
                    priceInfo={order.orderPriceInfo}
                    payment={order.payment}
                    isRefunding={isRefunding}
                />
            )}

            <OrderContacts
                className={cx('contacts')}
                email={order.contactInfo.email}
                phone={order.contactInfo.phone}
                info={i18nContacts.infoDotTrains()}
            />

            {service && <BusesRefundModal service={service} />}
        </OrderWrapper>
    );
};

export default memo(BusesOrder);
