import {FC, memo, useCallback, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {IBusesOrderTicket} from 'types/buses/order/IBusesOrderTicket';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';

import initRefundAction from 'reducers/account/genericOrder/refund/thunk/initRefundAction';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18n from 'i18n/buses-OrderPage-Passenger';
import * as i18nTrainPassenger from 'i18n/account-OrderTrains-Passenger';

import Button from 'components/Button/Button';
import Text from 'components/Text/Text';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import PassengerInfo from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/Passengers/components/Ticket/components/PassengerInfo/PassengerInfo';
import TicketInfo from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/Passengers/components/Ticket/components/TicketInfo/TicketInfo';
import Refund from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/Passengers/components/Ticket/components/Refund/Refund';
import Flex from 'components/Flex/Flex';

import cx from './Ticket.scss';

interface ITicketProps {
    ticket: IBusesOrderTicket;
    isRefundDisabled: boolean;
}

const Ticket: FC<ITicketProps> = props => {
    const {
        ticket: {passenger, seat, ticketType, price, refundPartInfo},
        isRefundDisabled,
    } = props;

    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const isRefunded = refundPartInfo?.state === ERefundPartState.REFUNDED;
    const refundContext = refundPartInfo?.context;

    const refundStatusBlock = useMemo(() => {
        if (!isRefunded) {
            return null;
        }

        return (
            <Text color="alert" size="s">
                {i18nTrainPassenger.statusDotRefund()}
            </Text>
        );
    }, [isRefunded]);

    const handleRefund = useCallback(() => {
        if (!refundContext) {
            return;
        }

        dispatch(initRefundAction([refundContext]));
    }, [dispatch, refundContext]);

    return (
        <CardWithDeviceLayout.Section
            className={cx('root', deviceMods('root', deviceType), {
                root_refunded: isRefunded,
            })}
            paddingTop={4}
            paddingBottom={4}
        >
            {deviceType.isMobile && refundStatusBlock}

            <Flex
                justifyContent="space-between"
                flexDirection={deviceType.isMobile ? 'column' : 'row'}
            >
                <div className={cx('passengerAndTicketInfo')}>
                    <PassengerInfo passenger={passenger} />

                    <TicketInfo
                        className={cx('ticketInfo')}
                        seat={seat}
                        ticketType={ticketType}
                        price={price}
                    />
                </div>

                {deviceType.isDesktop && refundStatusBlock}

                {refundPartInfo?.state === ERefundPartState.ENABLED && (
                    <Button
                        className={cx('refundButton')}
                        size={deviceType.isMobile ? 'm' : 's'}
                        disabled={isRefundDisabled}
                        onClick={handleRefund}
                    >
                        {i18n.refundTicket()}
                    </Button>
                )}
            </Flex>

            {refundPartInfo && (
                <Refund
                    className={cx('refund')}
                    refundPartInfo={refundPartInfo}
                />
            )}
        </CardWithDeviceLayout.Section>
    );
};

export default memo(Ticket);
