import {memo, FC, useMemo} from 'react';
import first from 'lodash/first';

import {IOrderPriceInfo} from 'server/api/HotelsBookAPI/types/IOrder';
import IPaymentInfo from 'server/api/GenericOrderApi/types/common/IPaymentInfo';
import IPrice from 'types/common/price/IPrice';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getReceipts from 'projects/account/pages/Order/components/GenericOrder/components/BusesOrder/components/Passengers/components/TotalInfo/utilities/getReceipts';

import * as i18nPrice from 'i18n/account-OrderTrains-Price';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import TotalPriceReceipts from 'projects/account/pages/Order/components/GenericOrder/components/TotalPriceReceipts/TotalPriceReceipts';

import cx from './TotalInfo.scss';

interface ITotalInfoProps {
    priceInfo: IOrderPriceInfo;
    payment: IPaymentInfo | null;
    totalRefundPrice: IPrice | null;
}

const TotalInfo: FC<ITotalInfoProps> = props => {
    const {priceInfo, payment, totalRefundPrice} = props;

    const deviceType = useDeviceType();

    const {paymentReceipts, refundReceipts} = useMemo(
        () => getReceipts(payment?.receipts),
        [payment?.receipts],
    );

    return (
        <CardWithDeviceLayout.Section
            className={cx(deviceMods('root', deviceType))}
            between={deviceType.isMobile ? 2 : 0}
        >
            <Text
                className={cx('totalPrice')}
                size={deviceType.isMobile ? 'm' : 'l'}
                weight="bold"
                tag="div"
            >
                {insertJSXIntoKey(i18nPrice.price)({
                    hasInsurance: false,
                    isDesktop: deviceType.isDesktop,
                    price: <Price {...priceInfo.price} />,
                })}
            </Text>

            {totalRefundPrice && (
                <Text className={cx('refund')} size="s" tag="div">
                    {insertJSXIntoKey(i18nPrice.refund)({
                        price: <Price {...totalRefundPrice} />,
                        isDesktop: deviceType.isDesktop,
                    })}
                </Text>
            )}

            <TotalPriceReceipts
                className={cx('receipts')}
                paymentReceipt={first(paymentReceipts)}
                refundReceipts={refundReceipts}
            />
        </CardWithDeviceLayout.Section>
    );
};

export default memo(TotalInfo);
