import {FC} from 'react';

import {IWithClassName} from 'types/withClassName';
import IReceiptItem from 'server/api/GenericOrderApi/types/common/IReceiptItem';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {INSURANCE_FULL_TERMS_LINK} from 'projects/account/lib/orders/links';

import * as i18nBlock from 'i18n/account-OrderTrains-Price';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import Intersperse from 'components/Intersperse/Intersperse';
import DotSeparator from 'components/DotSeparator/DotSeparator';

interface ITotalPriceReceiptsProps extends IWithClassName, IWithQaAttributes {
    paymentReceipt?: IReceiptItem;
    refundReceipts: IReceiptItem[];
    hasInsurance?: boolean;
    onPaymentReceiptClick?(): void;
}

const TotalPriceReceipts: FC<ITotalPriceReceiptsProps> = props => {
    const {
        className,
        paymentReceipt,
        refundReceipts,
        hasInsurance,
        onPaymentReceiptClick,
    } = props;
    const isSingleRefundReceipt = refundReceipts.length === 1;

    return (
        <Flex className={className} alignItems="center" flexWrap="wrap">
            <Intersperse separator={<DotSeparator />}>
                {paymentReceipt && (
                    <Link
                        url={paymentReceipt.url}
                        target="_blank"
                        rel="noopener noreferrer"
                        onClick={onPaymentReceiptClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'receipt',
                        })}
                    >
                        {i18nBlock.paymentReceipt()}
                    </Link>
                )}
                {refundReceipts.map((refundReceipt, receiptIndex) => (
                    <Link
                        key={receiptIndex}
                        url={refundReceipt.url}
                        target="_blank"
                        rel="noopener noreferrer"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'refundReceipt',
                        })}
                    >
                        {isSingleRefundReceipt
                            ? i18nBlock.singleRefundReceipt()
                            : i18nBlock.refundReceipt({
                                  receiptIndex: receiptIndex + 1,
                              })}
                    </Link>
                ))}
                {hasInsurance && (
                    <Link
                        url={INSURANCE_FULL_TERMS_LINK}
                        target="_blank"
                        rel="noopener noreferrer"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'insuranceDetails',
                        })}
                    >
                        {i18nBlock.insurance()}
                    </Link>
                )}
            </Intersperse>
        </Flex>
    );
};

export default TotalPriceReceipts;
