import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';

import * as actions from 'reducers/account/orders/actions';

import orderIdSelector from 'selectors/account/order/trains/orderIdSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Header from './components/Header/Header';
import Segments from './components/Segments/Segments';
import Warnings from './components/Warnings/Warnings';
import OrderActions from './components/OrderActions/OrderActions';
import Passengers from './components/Passengers/Passengers';
import Contacts from './components/Contacts/Contacts';
import RegistrationActionModal from './components/RegistrationActionModal/RegistrationActionModal';
import RefundActionModal from './components/RefundActionModal/RefundActionModal';

import useReloadAtOrder from './hooks/useReloadAtOrder';
import OrderWrapper from '../OrderWrapper/OrderWrapper';

import cx from './GenericOrderTrains.scss';

interface IGenericOrderTrainsProps extends IWithQaAttributes {}

const GenericOrderTrains: React.FC<IGenericOrderTrainsProps> = props => {
    const deviceType = useDeviceType();

    const rootQa = getQa(props);

    const orderId = useSelector(orderIdSelector);
    const dispatch = useDispatch();

    const updateOrder = useCallback(
        (isBackground: boolean) => {
            if (!orderId) {
                return;
            }

            dispatch(
                actions.updateOrder.request({
                    orderId,
                    trainsOrderType: ORDER_TYPE_VALUES.TRAINS_GENERIC,
                    isBackground,
                }),
            );
        },
        [dispatch, orderId],
    );

    useReloadAtOrder({updateOrder});

    const actionsNode = useMemo(() => {
        return (
            <OrderActions
                className={cx('orderActions')}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'orderActions',
                })}
            />
        );
    }, [rootQa]);

    return (
        <OrderWrapper
            className={cx(deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <Header
                className={cx('header')}
                {...prepareQaAttributes({parent: rootQa, current: 'header'})}
            />
            <Segments
                className={cx('segments')}
                actions={deviceType.isMobile && actionsNode}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'segmentsInfo',
                })}
            />
            <Warnings
                {...prepareQaAttributes({parent: rootQa, current: 'warnings'})}
            />
            {deviceType.isDesktop && actionsNode}
            <Passengers
                className={cx('passengers')}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'passengers',
                })}
            />
            <Contacts
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'contacts',
                })}
            />
            <RegistrationActionModal />
            <RefundActionModal />
        </OrderWrapper>
    );
};

export default GenericOrderTrains;
