import React, {useEffect} from 'react';

import {IWithDeviceType} from 'types/withDeviceType';

import {STATUS} from 'projects/account/lib/orders/actionsStatuses';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderTrains-Action';

import Modal from 'components/Modal/Modal';
import Heading from 'components/Heading/Heading';
import Box from 'components/Box/Box';
import Button from 'components/Button/Button';
import Card from 'components/Card/Card';
import BookLoader from 'components/BookLoader/BookLoader';

import cx from './OrderActionModal.scss';

interface IOrderActionModal extends IWithDeviceType, IWithQaAttributes {
    isVisible: boolean;
    submitName: string;
    title: string;
    description?: React.ReactNode;
    cancelName?: string;
    status: STATUS;
    onClose(): void;
    onSubmit(): void;
}

const OrderActionModal: React.FC<IOrderActionModal> = props => {
    const {
        deviceType,
        isVisible,
        onClose,
        onSubmit,
        cancelName = i18nBlock.cancel(),
        submitName,
        title,
        description,
        status,
    } = props;

    useEffect(() => {
        return (): void => {
            if (isVisible) {
                onClose();
            }
        };
    }, []);

    return (
        <Modal
            isVisible={isVisible}
            onClose={onClose}
            containerClassName={cx('container', {
                container_mobile: deviceType.isMobile,
            })}
        >
            <Modal.Content
                between="5"
                className={cx('modal', {
                    modal_mobile: deviceType.isMobile,
                })}
            >
                <div className={cx('block')}>
                    <Heading level="2" className={cx('title')}>
                        {title}
                    </Heading>
                </div>
                <div className={cx('description')}>{description}</div>
                <Box
                    className={cx('buttons')}
                    inline={deviceType.isDesktop}
                    between="2"
                >
                    <Button
                        onClick={onSubmit}
                        size="m-inset"
                        theme="secondary"
                        width="max"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'submitButton',
                        })}
                    >
                        {submitName}
                    </Button>
                    <Button
                        onClick={onClose}
                        size="m-inset"
                        type="button"
                        theme="primary"
                        width="max"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'cancelButton',
                        })}
                    >
                        {cancelName}
                    </Button>
                </Box>
                {status === STATUS.PENDING && (
                    <Card
                        className={cx('loader')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'loader',
                        })}
                    >
                        <BookLoader
                            title={i18nBlock.loading()}
                            isLoading={true}
                        />
                    </Card>
                )}
            </Modal.Content>
        </Modal>
    );
};

export default OrderActionModal;
