import React, {useCallback, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';
import IActionItem from 'projects/account/pages/Order/components/GenericOrderTrains/components/ActionsDialog/types/IActionItem';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import {isNotNull} from 'types/utilities';

import * as actions from 'reducers/account/orders/actions';

import orderActionsSelector from './selectors/orderActionsSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {ORDER_STATUS_VALUES} from 'projects/account/lib/orders/statuses';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import getDownloadBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getDownloadBlankUrl';
import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';
import {STATUS} from 'projects/account/lib/orders/actionsStatuses';
import getDownloadTicketButtonText from './utilities/getDownloadTicketButtonText';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/account-OrderTrains-Actions';

import CommonOrderActions, {
    EOrderActionsSource,
    IOrderActionsDownload,
    IOrderActionsPrint,
} from 'components/OrderActions/OrderActions';
import ActionsDialog from 'projects/account/pages/Order/components/GenericOrderTrains/components/ActionsDialog/ActionsDialog';
import useChangeRemoteCheckinActionItem from 'projects/account/pages/Order/components/GenericOrderTrains/hooks/useChangeRemoteCheckinActionItem';

import cx from './OrderActions.scss';

interface IOrderActionsProps extends IWithClassName, IWithQaAttributes {}

const OrderActions: React.FC<IOrderActionsProps> = props => {
    const {className} = props;
    const deviceType = useDeviceType();

    const {
        orderStatus,
        orderState,
        orderId,
        orderType,
        hasInsurance,
        notRefundedTickets,
        status,
        orderRefundPartInfo,
        orderRemoteCheckinAllowed,
        orderRemoteCheckinInfo,
    } = useSelector(orderActionsSelector);
    const dispatch = useDispatch();

    const {value: isOpened, setFalse: close, setTrue: open} = useBoolean(false);

    const moreRef = useRef<HTMLElement>(null);

    const renderIcon = useCallback(
        (
            _className?: string,
            defaultIcon?: JSX.Element,
        ): JSX.Element | undefined => {
            if (deviceType.isDesktop || !orderRemoteCheckinAllowed) {
                return defaultIcon;
            }

            return;
        },
        [deviceType, orderRemoteCheckinAllowed],
    );

    const printInfo = useMemo((): IOrderActionsPrint | undefined => {
        if (!orderId) {
            return undefined;
        }

        return {
            title: i18nBlock.print(),
            url: getDownloadBlankUrl({id: orderId}, orderType),
            renderIcon,
            onClick: (): void => {
                reachGoal(ETrainsGoal.ORDER_PRINT_TICKETS_CLICK);
            },
        };
    }, [orderId, orderType, renderIcon]);

    const downloadInfo = useMemo((): IOrderActionsDownload | undefined => {
        if (!orderId) {
            return undefined;
        }

        return {
            title: getDownloadTicketButtonText(deviceType, hasInsurance),
            fileName: getDownloadFilename('order', orderId, 'pdf'),
            url: getDownloadBlankUrl({id: orderId}, orderType),
            renderIcon,
            onClick: (): void => {
                reachGoal(ETrainsGoal.ORDER_DOWNLOAD_TICKETS_CLICK);
            },
        };
    }, [deviceType, hasInsurance, orderId, orderType, renderIcon]);

    const cancelInfo = useMemo(() => {
        if (
            !orderId ||
            !orderRefundPartInfo ||
            !orderRefundPartInfo.context ||
            orderRefundPartInfo.state !== ERefundPartState.ENABLED
        ) {
            return undefined;
        }

        return {
            title:
                notRefundedTickets.length === 1
                    ? i18nBlock.refundTicket()
                    : i18nBlock.refundAllTickets(),
            isDisabled: status === STATUS.PENDING,
            tooltipTitle: i18nBlock.waitingForRefund(),
            renderIcon,
            onClick: (): void => {
                dispatch(
                    actions.setSelectedTickets(
                        notRefundedTickets.map(ticket =>
                            String(ticket.blankId),
                        ),
                    ),
                );
                dispatch(
                    actions.orderRefundAmountRequest({
                        orderId,
                        refundPartContexts: [orderRefundPartInfo.context],
                    }),
                );

                reachGoal(ETrainsGoal.ORDER_REFUND_ALL_TICKETS_CLICK);
            },
        };
    }, [
        orderId,
        orderRefundPartInfo,
        notRefundedTickets,
        status,
        renderIcon,
        dispatch,
    ]);

    const moreInfo = useMemo(() => {
        if (!orderRemoteCheckinAllowed) {
            return;
        }

        return {
            isDisabled: status === STATUS.PENDING,
            innerRef: moreRef,
            onClick: open,
        };
    }, [orderRemoteCheckinAllowed, open, moreRef, status]);

    const changeRemoteCheckinDialogAction = useChangeRemoteCheckinActionItem(
        orderRemoteCheckinInfo,
    );

    const actionItems: IActionItem[] = useMemo(() => {
        return [changeRemoteCheckinDialogAction].filter(isNotNull);
    }, [changeRemoteCheckinDialogAction]);

    if (
        orderStatus !== ORDER_STATUS_VALUES.DONE ||
        orderState === EGenericOrderState.REFUNDED
    ) {
        return null;
    }

    return (
        <div className={cx(className, 'root', deviceMods('root', deviceType))}>
            <div className={cx('actions')}>
                <CommonOrderActions
                    className={cx('actions')}
                    source={EOrderActionsSource.ORDER_PAGE}
                    print={printInfo}
                    download={downloadInfo}
                    cancel={cancelInfo}
                    more={moreInfo}
                    {...prepareQaAttributes(props)}
                />
            </div>
            <ActionsDialog
                actions={actionItems}
                isOpened={isOpened}
                buttonRef={moreRef}
                onClose={close}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'actionsDialog',
                })}
            />
        </div>
    );
};

export default OrderActions;
