import {createSelector} from 'reselect';
import flatMap from 'lodash/flatMap';
import uniq from 'lodash/uniq';

import {isNotUndefined} from 'types/utilities';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';
import {ORDER_TRAINS_REGISTRATION_STATUS} from 'projects/account/pages/Order/types/OrderTrainsRegistrationStatus';

import trainServicesSelector from 'selectors/account/order/trains/trainServicesSelector';

import toggleRegistrationStatusByRzhdStatus from 'projects/account/pages/Order/components/GenericOrderTrains/utilities/toggleRegistrationStatusByRzhdStatus';

export interface IRemoteCheckinInfo {
    newStatus: ORDER_TRAINS_REGISTRATION_STATUS;
    ticketIds: number[];
}

const orderRemoteCheckinInfoSelector = createSelector(
    [trainServicesSelector],
    (services): IRemoteCheckinInfo | null => {
        const passengers = flatMap(
            services,
            service => service.trainInfo.passengers,
        ).filter(
            passenger =>
                passenger.refundPartInfo?.state === ERefundPartState.ENABLED,
        );
        const ticketIds = passengers
            .map(passenger => passenger.ticket?.blankId)
            .filter(isNotUndefined);

        if (!passengers.length) {
            return null;
        }

        const uniqTicketStatuses = uniq(
            passengers.map(passenger => passenger?.ticket?.rzhdStatus),
        );

        /**
         * Если в заказе есть билеты с разными rzhdStatus
         * то мы не применяем над таким заказом общих действий с ЭР
         */
        if (uniqTicketStatuses.length > 1) {
            return null;
        }

        const status = uniqTicketStatuses[0];

        if (!status) {
            return null;
        }

        const newStatus = toggleRegistrationStatusByRzhdStatus(status);

        if (!newStatus) {
            return null;
        }

        return {
            newStatus,
            ticketIds,
        };
    },
);

export default orderRemoteCheckinInfoSelector;
