import React from 'react';

import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';

import {IPassengerWithTickets} from 'projects/trains/lib/order/getPassengerWithTicketsListByGenericOrderServices';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import PassengerInfo from './components/PassengerInfo/PassengerInfo';
import PassengerTickets from './components/PassengerTickets/PassengerTickets';
import SpaceSeparator from 'components/SpaceSeparator/SpaceSeparator';
import PassengerRefundWrapper from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerRefundWrapper/PassengerRefundWrapper';
import Flex from 'components/Flex/Flex';
import PassengerTicketStatus from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketStatus/PassengerTicketStatus';
import PassengerTicketActions from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketActions/PassengerTicketActions';

import cx from './Passenger.scss';

interface IPassengerProps extends IWithQaAttributes {
    passengerWithTickets: IPassengerWithTickets;
}

const Passenger: React.FC<IPassengerProps> = props => {
    const {
        passengerWithTickets: {passenger, baby, tickets},
    } = props;

    const deviceType = useDeviceType();

    const passengerHasMultipleTickets = tickets.length > 1;
    const allTicketsRefunded = tickets.every(ticket => {
        return ticket.refundPartInfo?.state === ERefundPartState.REFUNDED;
    });
    const firstTicketWithTrainInfo = tickets[0];
    const ticket = firstTicketWithTrainInfo?.ticket;
    const refundPartInfo = firstTicketWithTrainInfo?.refundPartInfo;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            {!passengerHasMultipleTickets && ticket && deviceType.isMobile && (
                <PassengerTicketStatus
                    className={cx('status')}
                    ticket={ticket}
                    refundPartInfo={refundPartInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'refundTicketStatus',
                    })}
                />
            )}

            <Flex justifyContent="space-between" alignItems="baseline">
                <PassengerRefundWrapper
                    className={cx('passengerInfo')}
                    refunded={allTicketsRefunded}
                >
                    <PassengerInfo
                        passenger={passenger}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'info',
                        })}
                    />

                    {baby && (
                        <PassengerInfo
                            className={cx('babyInfo')}
                            passenger={baby}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'babyInfo',
                            })}
                        />
                    )}
                </PassengerRefundWrapper>

                {!passengerHasMultipleTickets && ticket && (
                    <>
                        {deviceType.isDesktop && (
                            <>
                                <PassengerTicketStatus
                                    className={cx('status')}
                                    ticket={ticket}
                                    refundPartInfo={refundPartInfo}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'refundTicketStatus',
                                    })}
                                />

                                <PassengerTicketActions
                                    ticket={ticket}
                                    refundPartInfo={refundPartInfo}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'actions',
                                    })}
                                />
                            </>
                        )}
                    </>
                )}
            </Flex>

            {passengerHasMultipleTickets ? (
                <Separator margin={3} />
            ) : (
                <SpaceSeparator space={3} />
            )}

            <Intersperse separator={<Separator />}>
                <PassengerTickets
                    tickets={tickets}
                    passengerHasMultipleTickets={passengerHasMultipleTickets}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'tickets',
                    })}
                />
            </Intersperse>
        </div>
    );
};

export default Passenger;
