import React from 'react';
import moment from 'moment-timezone';

import {MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON} from 'projects/trains/constants/documentTypes';

import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getDocumentTitle} from 'projects/trains/lib/order/fields/utils';
import {formatDocumentNumber} from 'utilities/documents/formatDocumentNumber';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {getPassengerGenderTitle} from 'projects/trains/lib/order/gender/getPassengerGenderTitle';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getPassengerFullName from 'projects/account/pages/Order/components/GenericOrderTrains/utilities/getPassengerFullName';

import * as i18nBlock from 'i18n/account-OrderTrains-Passenger';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import Intersperse from 'components/Intersperse/Intersperse';

import cx from './PassengerInfo.scss';

interface IPassengerInfoProps extends IWithClassName, IWithQaAttributes {
    passenger: ITrainPassenger;
}

const PassengerInfo: React.FC<IPassengerInfoProps> = props => {
    const {className, passenger} = props;
    const {docId, sex, docType, birthDate} = passenger;

    const deviceType = useDeviceType();

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            flexDirection="column"
            {...prepareQaAttributes(props)}
        >
            <Text
                size={deviceType.isDesktop ? 'l' : 'm'}
                weight="bold"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'name',
                })}
            >
                {getPassengerFullName(passenger)}
            </Text>

            <Text
                className={cx('info')}
                size={deviceType.isDesktop ? 's' : 'm'}
            >
                <Intersperse
                    separator={deviceType.isDesktop ? <DotSeparator /> : <br />}
                >
                    {docId && (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'document',
                            })}
                        >
                            {`${getDocumentTitle(
                                docType,
                            )}: ${formatDocumentNumber(
                                docId,
                                MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON[docType],
                            )}`}
                        </span>
                    )}
                    {birthDate && (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'birthDate',
                            })}
                        >
                            {i18nBlock.birthDate({
                                date: moment
                                    .parseZone(birthDate)
                                    .format(HUMAN_DATE_RU),
                            })}
                        </span>
                    )}
                    {sex && (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'gender',
                            })}
                        >
                            {getPassengerGenderTitle(sex)}
                        </span>
                    )}
                </Intersperse>
            </Text>
        </Flex>
    );
};

export default PassengerInfo;
