import React, {useMemo} from 'react';

import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';

import {ITicketWithTrainInfo} from 'projects/trains/lib/order/getPassengerWithTicketsListByGenericOrderServices';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_EM_DASH} from 'utilities/strings/charCodes';
import {deviceMods} from 'utilities/stylesUtils';
import getFromToByStations from 'projects/trains/lib/getFromToByStations';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import Text, {TTextSize} from 'components/Text/Text';
import PassengerTicketStatus from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketStatus/PassengerTicketStatus';
import PassengerTicketRefund from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketRefund/PassengerTicketRefund';
import PassengerTicketActions from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketActions/PassengerTicketActions';
import PassengerRefundWrapper from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerRefundWrapper/PassengerRefundWrapper';

import PassengerTicketPlaces from '../PassengerTicketPlaces/PassengerTicketPlaces';
import PassengerTicketTariff from '../PassengerTicketTariff/PassengerTicketTariff';
import PassengerTicketInsurance from '../PassengerTicketInsurance/PassengerTicketInsurance';
import PassengerTicketRemoteCheckinStatus from '../PassengerTicketRemoteCheckinStatus/PassengerTicketRemoteCheckinStatus';

import cx from './PassengerTicket.scss';

interface IPassengerTicketProps extends IWithQaAttributes {
    ticketWithTrainInfo: ITicketWithTrainInfo;
    passengerHasMultipleTickets: boolean;
}

const PassengerTicket: React.FC<IPassengerTicketProps> = props => {
    const {ticketWithTrainInfo, passengerHasMultipleTickets} = props;
    const {
        ticket,
        refundPartInfo,
        trainInfo: {compartmentGender, insuranceStatus, stationFrom, stationTo},
        nonRefundableTariff,
        insurance,
        withBaby,
        babyInsurance,
        babyRefundPartInfo,
    } = ticketWithTrainInfo;

    const deviceType = useDeviceType();

    const title = useMemo(() => {
        const {from, to} = getFromToByStations({stationFrom, stationTo});

        return `${from} ${CHAR_EM_DASH} ${to}`;
    }, [stationFrom, stationTo]);

    if (!ticket) {
        return null;
    }

    const ticketRefunded = refundPartInfo?.state === ERefundPartState.REFUNDED;

    const textSize: TTextSize = deviceType.isDesktop ? 's' : 'm';

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType))}
            flexDirection="column"
            {...prepareQaAttributes(props)}
        >
            {passengerHasMultipleTickets && (
                <>
                    {deviceType.isMobile && (
                        <PassengerTicketStatus
                            className={cx('status')}
                            ticket={ticket}
                            refundPartInfo={refundPartInfo}
                        />
                    )}

                    <Flex justifyContent="space-between" alignItems="baseline">
                        <PassengerRefundWrapper
                            className={cx('title')}
                            refunded={ticketRefunded}
                        >
                            <Text
                                size={textSize}
                                weight={
                                    deviceType.isMobile ? undefined : 'bold'
                                }
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'routeTitle',
                                })}
                            >
                                {title}
                            </Text>
                        </PassengerRefundWrapper>

                        {deviceType.isDesktop && (
                            <PassengerTicketStatus
                                className={cx('status')}
                                ticket={ticket}
                                refundPartInfo={refundPartInfo}
                            />
                        )}

                        {deviceType.isDesktop && (
                            <PassengerTicketActions
                                ticket={ticket}
                                refundPartInfo={refundPartInfo}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'actions',
                                })}
                            />
                        )}
                    </Flex>
                </>
            )}

            <PassengerRefundWrapper refunded={ticketRefunded}>
                <Text size={textSize}>
                    <PassengerTicketPlaces
                        ticket={ticket}
                        compartmentGender={compartmentGender}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'places',
                        })}
                    />
                </Text>

                <Flex
                    flexDirection={deviceType.isMobile ? 'column' : 'row'}
                    inline={deviceType.isDesktop}
                    between={deviceType.isMobile ? undefined : 5}
                >
                    <PassengerTicketTariff
                        ticket={ticket}
                        nonRefundableTariff={nonRefundableTariff}
                        withBaby={withBaby}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'tariff',
                        })}
                    />

                    <PassengerTicketInsurance
                        insurance={insurance}
                        babyInsurance={babyInsurance}
                        insuranceStatus={insuranceStatus}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'insurance',
                        })}
                    />
                </Flex>

                <PassengerTicketRemoteCheckinStatus
                    className={cx('remoteCheckinStatus')}
                    ticket={ticket}
                    refundPartInfo={refundPartInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'remoteCheckinStatus',
                    })}
                />
            </PassengerRefundWrapper>

            {deviceType.isMobile && (
                <PassengerTicketActions
                    className={cx('actions')}
                    ticket={ticket}
                    refundPartInfo={refundPartInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'actions',
                    })}
                />
            )}

            <PassengerTicketRefund
                className={cx('refund')}
                ticket={ticket}
                refundPartInfo={refundPartInfo}
                babyRefundPartInfo={babyRefundPartInfo}
                {...prepareQaAttributes({parent: props, current: 'refund'})}
            />
        </Flex>
    );
};

export default PassengerTicket;
