import React, {useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import {IRefundPartInfo} from 'server/api/GenericOrderApi/types/common/refund/IRefundPartInfo';
import IActionItem from 'projects/account/pages/Order/components/GenericOrderTrains/components/ActionsDialog/types/IActionItem';
import EActionType from 'projects/account/pages/Order/components/GenericOrderTrains/components/ActionsDialog/types/EActionType';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {isNotNull} from 'types/utilities';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';

import * as actions from 'reducers/account/orders/actions';

import actionsStatusSelector from 'selectors/account/order/trains/actionsStatusSelector';
import orderIdSelector from 'selectors/account/order/trains/orderIdSelector';
import orderStateSelector from 'selectors/account/order/trains/orderStateSelector';

import {STATUS} from 'projects/account/lib/orders/actionsStatuses';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import useChangeRemoteCheckinActionItem from 'projects/account/pages/Order/components/GenericOrderTrains/hooks/useChangeRemoteCheckinActionItem';
import Actions from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketActions/components/Actions/Actions';

import useRemoteCheckinInfo from './hooks/useRemoteCheckinInfo';

interface IPassengerTicketActionsProps
    extends IWithClassName,
        IWithQaAttributes {
    ticket: ITrainTicket;
    refundPartInfo: IRefundPartInfo | null;
}

const PassengerTicketActions: React.FC<IPassengerTicketActionsProps> =
    props => {
        const {className, ticket, refundPartInfo} = props;

        const orderId = useSelector(orderIdSelector);
        const status = useSelector(actionsStatusSelector);
        const orderState = useSelector(orderStateSelector);
        const dispatch = useDispatch();

        const remoteCheckinInfo = useRemoteCheckinInfo(ticket, refundPartInfo);
        const changeRemoteCheckinActionItem =
            useChangeRemoteCheckinActionItem(remoteCheckinInfo);

        const refundActionItem = useMemo((): IActionItem | null => {
            if (
                !orderId ||
                !refundPartInfo ||
                !refundPartInfo.context ||
                refundPartInfo.state !== ERefundPartState.ENABLED
            ) {
                return null;
            }

            return {
                type: EActionType.REFUND_TICKET,
                disabled:
                    status === STATUS.PENDING ||
                    orderState === EGenericOrderState.WAITING_REFUND,
                onClick: (): void => {
                    dispatch(
                        actions.setSelectedTickets([String(ticket.blankId)]),
                    );
                    dispatch(
                        actions.orderRefundAmountRequest({
                            orderId,
                            refundPartContexts: [refundPartInfo.context],
                        }),
                    );

                    reachGoal(ETrainsGoal.ORDER_REFUND_TICKET_CLICK);
                },
            };
        }, [orderState, ticket, refundPartInfo, status, orderId, dispatch]);

        const otherActionItems: IActionItem[] = useMemo(() => {
            return [changeRemoteCheckinActionItem].filter(isNotNull);
        }, [changeRemoteCheckinActionItem]);

        return (
            <Actions
                className={className}
                refundActionItem={refundActionItem}
                otherActionItems={otherActionItems}
                {...prepareQaAttributes(props)}
            />
        );
    };

export default PassengerTicketActions;
