import React from 'react';

import ITrainInsurance from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainInsurance';
import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {INSURANCE_FULL_TERMS_LINK} from 'projects/account/lib/orders/links';

import * as i18nAccountPassenger from 'i18n/account-OrderTrains-Passenger';
import * as i18nAccountPassengers from 'i18n/account-OrderTrains-Passengers';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Link from 'components/Link/Link';
import InfoPopup from 'components/InfoPopup/InfoPopup';
import Flex from 'components/Flex/Flex';
import Box from 'components/Box/Box';

import cx from './PassengerTicketInsurance.scss';

interface IPassengerTicketInsuranceProps extends IWithQaAttributes {
    insurance: ITrainInsurance | null;
    babyInsurance: ITrainInsurance | null;
    insuranceStatus: TrainsInsuranceStatus;
}

const PassengerTicketInsurance: React.FC<IPassengerTicketInsuranceProps> =
    props => {
        const {insurance, babyInsurance, insuranceStatus} = props;

        const deviceType = useDeviceType();

        if (
            !insurance ||
            insuranceStatus !== TrainsInsuranceStatus.CHECKED_OUT
        ) {
            return null;
        }

        return (
            <Flex
                flexDirection="row"
                alignItems="center"
                between={2}
                inline
                {...prepareQaAttributes(props)}
            >
                <Text size={deviceType.isDesktop ? 's' : 'm'} color="secondary">
                    {insertJSXIntoKey(i18nAccountPassenger.insurance)({
                        price: (
                            <Price
                                value={
                                    insurance.amount.value +
                                    (babyInsurance?.amount.value || 0)
                                }
                                currency={insurance.amount.currency}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'price',
                                })}
                            />
                        ),
                    })}
                </Text>

                <InfoPopup
                    iconClassName={cx('icon')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'questionButton',
                    })}
                >
                    <Box between={4}>
                        {babyInsurance && (
                            <div>
                                <div>
                                    {insertJSXIntoKey(
                                        i18nAccountPassengers.insuranceWithPrice,
                                    )({
                                        title: i18nAccountPassengers.insuranceAdultTitle(),
                                        price: (
                                            <Price
                                                value={insurance.amount.value}
                                                currency={
                                                    insurance.amount.currency
                                                }
                                            />
                                        ),
                                    })}
                                </div>

                                <div>
                                    {insertJSXIntoKey(
                                        i18nAccountPassengers.insuranceWithPrice,
                                    )({
                                        title: i18nAccountPassengers.insuranceBabyTitle(),
                                        price: (
                                            <Price
                                                value={
                                                    babyInsurance.amount.value
                                                }
                                                currency={
                                                    babyInsurance.amount
                                                        .currency
                                                }
                                            />
                                        ),
                                    })}
                                </div>
                            </div>
                        )}

                        <div>
                            {i18nAccountPassengers.insurance()}{' '}
                            <Link
                                url={INSURANCE_FULL_TERMS_LINK}
                                target="_blank"
                                rel="noopener noreferrer"
                                theme={
                                    deviceType.isDesktop ? 'brand' : 'normal'
                                }
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'detailsLink',
                                })}
                            >
                                {i18nAccountPassengers.link()}
                            </Link>
                        </div>
                    </Box>
                </InfoPopup>
            </Flex>
        );
    };

export default PassengerTicketInsurance;
