import React from 'react';
import {useSelector} from 'react-redux';

import {TrainsOrderInfoTicketRzhdStatus} from 'server/api/TrainsBookingApi/types/TrainsOrderInfoTicketRzhdStatus';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import {IRefundPartInfo} from 'server/api/GenericOrderApi/types/common/refund/IRefundPartInfo';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import orderIdSelector from 'selectors/account/order/trains/orderIdSelector';
import {trainsOrderTypeSelector} from 'selectors/account/ordersSelector';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import getTicketRefundBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getTicketRefundBlankUrl';
import getTotalRefundPrice from 'projects/account/pages/Order/components/GenericOrderTrains/components/Passenger/components/PassengerTicketRefund/components/PassengerTicketRefundContent/utilities/getTotalRefundPrice';

import * as i18nPriceBlock from 'i18n/account-OrderTrains-Price';
import * as i18nBlock from 'i18n/account-OrderTrains-Passenger';

import Price from 'components/Price/Price';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import Intersperse from 'components/Intersperse/Intersperse';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import DownloadReceiptLink from 'projects/account/pages/Order/components/DownloadReceiptLink/DownloadReceiptLink';
import Flex from 'components/Flex/Flex';

interface IPassengerTicketRefundContentProps extends IWithQaAttributes {
    ticket: ITrainTicket;
    refundPartInfo: IRefundPartInfo;
    babyRefundPartInfo: IRefundPartInfo | null;
}

const PassengerTicketRefundContent: React.FC<IPassengerTicketRefundContentProps> =
    props => {
        const {ticket, refundPartInfo, babyRefundPartInfo} = props;

        const deviceType = useDeviceType();

        const orderId = useSelector(orderIdSelector);
        const orderType = useSelector(trainsOrderTypeSelector);

        const handleReceiptClick = useReachGoal(
            ETrainsGoal.ORDER_DOWNLOAD_REFUND_RECEIPT_CLICK,
        );

        if (
            ticket.rzhdStatus ===
            TrainsOrderInfoTicketRzhdStatus.PLACES_REFUNDED
        ) {
            return <div>{i18nBlock.refundDotLongText()}</div>;
        }

        if (!orderId) {
            return null;
        }

        const receiptUrl =
            refundPartInfo?.refund?.paymentRefundReceiptUrls?.[0];

        const refundPrice = getTotalRefundPrice(
            refundPartInfo,
            babyRefundPartInfo,
        );

        const blankUrl = getTicketRefundBlankUrl({
            orderId,
            orderType,
            blankId: String(ticket.blankId),
            refundPartInfo,
        });

        return (
            <Flex
                flexDirection={deviceType.isDesktop ? 'row' : 'column'}
                inline={deviceType.isDesktop}
                between={deviceType.isDesktop ? 5 : undefined}
            >
                {refundPrice && (
                    <Text size="s">
                        {insertJSXIntoKey(i18nPriceBlock.refund)({
                            isDesktop: true,
                            price: (
                                <Text weight="bold" size="s">
                                    <Price
                                        {...refundPrice}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'price',
                                        })}
                                    />
                                </Text>
                            ),
                        })}
                    </Text>
                )}
                <div>
                    <Intersperse separator={<DotSeparator />}>
                        {blankUrl && (
                            <Link
                                url={blankUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'downloadTicket',
                                })}
                            >
                                {i18nBlock.refundDotTicket()}
                            </Link>
                        )}
                        <DownloadReceiptLink
                            receiptUrl={receiptUrl}
                            onClick={handleReceiptClick}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'downloadReceipt',
                            })}
                        />
                    </Intersperse>
                </div>
            </Flex>
        );
    };

export default PassengerTicketRefundContent;
