import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import {TrainsOrderInfoTicketRzhdStatus} from 'server/api/TrainsBookingApi/types/TrainsOrderInfoTicketRzhdStatus';
import {IRefundPartInfo} from 'server/api/GenericOrderApi/types/common/refund/IRefundPartInfo';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';

import orderStateSelector from 'selectors/account/order/trains/orderStateSelector';

import {OFFLINE_REFUND_LINK} from 'projects/account/lib/orders/links';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18n from 'i18n/account-OrderTrains-Passenger';

import Link from 'components/Link/Link';
import Text from 'components/Text/Text';

interface IPassengerTicketStatusProps
    extends IWithClassName,
        IWithQaAttributes {
    ticket: ITrainTicket;
    refundPartInfo: IRefundPartInfo | null;
}

const PassengerTicketStatus: React.FC<IPassengerTicketStatusProps> = props => {
    const {ticket, refundPartInfo, className} = props;
    const orderState = useSelector(orderStateSelector);

    const status = useMemo((): React.ReactNode => {
        if (
            orderState === EGenericOrderState.WAITING_REFUND ||
            orderState === EGenericOrderState.CANCELLED
        ) {
            return null;
        }

        if (
            ticket.rzhdStatus ===
            TrainsOrderInfoTicketRzhdStatus.PLACES_REFUNDED
        ) {
            return i18n.statusDotRefundPlace();
        }

        if (
            ticket.rzhdStatus ===
            TrainsOrderInfoTicketRzhdStatus.STRICT_BOARDING_PASS
        ) {
            return i18n.statusDotTakenAway();
        }

        if (ticket.rzhdStatus === TrainsOrderInfoTicketRzhdStatus.REFUNDED) {
            return i18n.statusDotRefund();
        }

        const refundState = refundPartInfo?.state;

        if (refundState === ERefundPartState.OFFLINE_ENABLED) {
            return (
                <Link
                    url={OFFLINE_REFUND_LINK}
                    target="_blank"
                    rel="noopener noreferrer"
                    theme="normal"
                >
                    {i18n.offlineRefund()}
                </Link>
            );
        }

        return null;
    }, [orderState, ticket, refundPartInfo]);

    if (!status) {
        return null;
    }

    return (
        <Text
            className={className}
            size="s"
            color="alert"
            {...prepareQaAttributes(props)}
        >
            {status}
        </Text>
    );
};

export default PassengerTicketStatus;
