import React from 'react';

import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import {IWithClassName} from 'types/withClassName';

import {CHAR_EM_DASH, CHAR_SPACE} from 'utilities/strings/charCodes';
import getTrainBookedTariffText from 'projects/trains/lib/order/getTrainBookedTariffText';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getTariffTextWithBaby from 'projects/trains/lib/order/tariffs/getTariffTextWithBaby';

import * as i18nAccountOrderBlock from 'i18n/account-Order';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import PriceExplanation from 'projects/trains/components/TrainsOrderPage/PriceExplanation/PriceExplanation';

import cx from './PassengerTicketTariff.scss';

interface IPassengerTicketTariffProps
    extends IWithClassName,
        IWithQaAttributes {
    ticket: ITrainTicket;
    nonRefundableTariff: boolean;
    withBaby: boolean;
}

const PassengerTicketTariff: React.FC<IPassengerTicketTariffProps> = props => {
    const {className, ticket, nonRefundableTariff, withBaby} = props;
    const {amount, bookedTariffCode, payment, rawTariffTitle} = ticket;

    const deviceType = useDeviceType();

    const tariffText = bookedTariffCode
        ? getTariffTextWithBaby(
              getTrainBookedTariffText(bookedTariffCode),
              withBaby,
          )
        : rawTariffTitle;
    const tariffTextWithRefundableInfo = nonRefundableTariff
        ? i18nAccountOrderBlock.nonRefundableTariff({tariff: tariffText})
        : tariffText;

    return (
        <div className={className}>
            <Text size={deviceType.isDesktop ? 's' : 'm'} color="secondary">
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'tariffName',
                    })}
                >
                    {tariffTextWithRefundableInfo}
                </span>
                {CHAR_SPACE}
                {CHAR_EM_DASH}
                {CHAR_SPACE}
                <Price
                    value={amount.value}
                    currency={amount.currency}
                    {...prepareQaAttributes({parent: props, current: 'price'})}
                />
            </Text>
            {amount.value !== 0 && (
                <PriceExplanation
                    className={cx('priceExplanation')}
                    iconClassName={cx('icon')}
                    ticketsPrice={payment.amount}
                    feePrice={payment.fee}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'priceExplanation',
                    })}
                />
            )}
        </div>
    );
};

export default PassengerTicketTariff;
