import React, {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {batchActions} from 'redux-batched-actions';

import * as actions from 'reducers/account/orders/actions';

import refundActionModalSelector from './selectors/refundActionModalSelector';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/account-OrderTrains-Action';
import * as i18nAccountOrderBlock from 'i18n/account-Order';

import RefundDescription from './components/RefundDescription/RefundDescription';

import OrderActionModal from '../OrderActionModal/OrderActionModal';

const RefundActionModal: React.FC = () => {
    const deviceType = useDeviceType();
    const {
        orderId,
        refundToken,
        totalAmount,
        refundAmount,
        actionsStatuses: {refundStatus},
        isRefundModalOpened,
        hasNonRefundableTariff,
        selectedTicketIds,
        selectedPassengerNames,
        selectedTicketFromTo,
        isMultipleTrainsOrder,
    } = useSelector(refundActionModalSelector);
    const dispatch = useDispatch();
    const handleSubmit = useCallback((): void => {
        if (!orderId || !refundToken) {
            return;
        }

        dispatch(
            actions.orderRefund.request({
                orderId,
                refundToken,
            }),
        );
    }, [dispatch, orderId, refundToken]);

    const handleClose = useCallback((): void => {
        dispatch(
            batchActions([
                actions.setSelectedTickets([]),
                actions.setRefundModalOpenedState(false),
            ]),
        );
    }, [dispatch]);

    const isMultipleTicketOrder = selectedTicketIds.length > 1;

    return (
        <OrderActionModal
            isVisible={isRefundModalOpened}
            deviceType={deviceType}
            onClose={handleClose}
            onSubmit={handleSubmit}
            status={refundStatus}
            title={
                isMultipleTicketOrder
                    ? i18nBlock.refundSomeHeader()
                    : i18nBlock.refundDotHeader()
            }
            description={
                <RefundDescription
                    hasNonRefundableTariff={hasNonRefundableTariff}
                    totalAmount={totalAmount}
                    refundAmount={refundAmount}
                    selectedPassengerNames={selectedPassengerNames}
                    selectedTicketFromTo={selectedTicketFromTo}
                    isMultipleTrainsOrder={isMultipleTrainsOrder}
                />
            }
            submitName={
                isMultipleTicketOrder
                    ? i18nAccountOrderBlock.trainsDotActionsDotRefundModalDotRefundTickets()
                    : i18nAccountOrderBlock.trainsDotActionsDotRefundModalDotRefundTicket()
            }
            cancelName={i18nAccountOrderBlock.trainsDotActionsDotRefundModalDotCancel()}
            {...prepareQaAttributes('orderTrainsOrchActionModal')}
        />
    );
};

export default RefundActionModal;
