import React, {useMemo} from 'react';

import IPrice from 'utilities/currency/PriceInterface';
import {CHAR_WIDE_MIDDLE_DASH} from 'utilities/strings/charCodes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nAccountOrderBlock from 'i18n/account-Order';
import * as i18nBlock from 'i18n/account-OrderTrains-Action';

import Price from 'components/Price/Price';
import Box from 'components/Box/Box';
import RefundWarning from 'projects/account/pages/Order/components/GenericOrderTrains/components/RefundActionModal/components/RefundWarning/RefundWarning';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';

interface IRefundDescriptionProps {
    hasNonRefundableTariff: boolean;
    refundAmount: IPrice | null;
    totalAmount: IPrice | null;
    selectedPassengerNames: string[] | null;
    selectedTicketFromTo: string | null;
    isMultipleTrainsOrder: boolean;
}

const RefundDescription: React.FC<IRefundDescriptionProps> = props => {
    const {
        hasNonRefundableTariff,
        refundAmount,
        totalAmount,
        selectedPassengerNames,
        selectedTicketFromTo,
        isMultipleTrainsOrder,
    } = props;

    const refundText = useMemo(() => {
        if (!refundAmount?.value && hasNonRefundableTariff) {
            return i18nBlock.refundNonRefundableTariffText();
        }

        if (!refundAmount || !totalAmount) {
            return null;
        }

        return insertJSXIntoKey(
            i18nAccountOrderBlock.trainsDotActionsDotRefundModalDotRefundAndTotal,
        )({
            separator: CHAR_WIDE_MIDDLE_DASH,
            refundPriceNode: (
                <Text weight="bold">
                    <Price
                        {...prepareQaAttributes({
                            parent: 'orderTrainsOrchActionModal',
                            current: 'price',
                        })}
                        {...refundAmount}
                    />
                </Text>
            ),
            totalPriceNode: (
                <Text weight="bold">
                    <Price
                        {...prepareQaAttributes({
                            parent: 'orderTrainsOrchActionModal',
                            current: 'totalPrice',
                        })}
                        {...totalAmount}
                    />
                </Text>
            ),
        });
    }, [refundAmount, totalAmount, hasNonRefundableTariff]);

    return (
        <Flex flexDirection="column" between={2}>
            {isMultipleTrainsOrder && (
                <Text size="m" weight="bold" tag="div">
                    {selectedTicketFromTo}
                </Text>
            )}
            {selectedPassengerNames && (
                <div>{selectedPassengerNames.join(', ')}</div>
            )}
            {refundText && <div>{refundText}</div>}
            <Box tag="div">
                <Text tag="div" color="alert">
                    <RefundWarning />
                    <div>{i18nBlock.refundDotWarning()}</div>
                </Text>
            </Box>
        </Flex>
    );
};

export default RefundDescription;
