import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import partnerReservationNumbersSelector from 'selectors/account/order/trains/partnerReservationNumbersSelector';
import trainsOrderSegmentsInfoByGenericServicesSelector from 'selectors/account/order/trains/trainsOrderSegmentsInfoByGenericServicesSelector';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderTrains-Info';

import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import TrainsOrderSegments from 'projects/trains/components/TrainsOrderSegments/TrainsOrderSegments';
import Box from 'components/Box/Box';

interface ISegmentsProps extends IWithClassName, IWithQaAttributes {
    actions?: React.ReactNode;
}

const Segments: React.FC<ISegmentsProps> = props => {
    const {className, actions} = props;

    const partnerReservationNumbersQa = getQa({
        parent: props,
        current: 'partnerReservationNumbers',
    });

    const partnerReservationNumbers = useSelector(
        partnerReservationNumbersSelector,
    );
    const {tripInfo, segments} = useSelector(
        trainsOrderSegmentsInfoByGenericServicesSelector,
    );

    const footerNode = useMemo(() => {
        if (!partnerReservationNumbers.length) {
            return null;
        }

        const numbers = partnerReservationNumbers.join(', ');

        return (
            <>
                {actions && (
                    <Box above={4} below={4} x={4}>
                        {actions}
                    </Box>
                )}
                <Separator />
                <CardWithDeviceLayout.Section paddingTop={3} paddingBottom={3}>
                    <Text
                        size="s"
                        color="secondary"
                        {...prepareQaAttributes(partnerReservationNumbersQa)}
                    >
                        {partnerReservationNumbers.length > 1
                            ? i18nBlock.partnerOrderNumbers({numbers})
                            : i18nBlock.partnerOrderNumber({
                                  number: numbers,
                              })}
                    </Text>
                </CardWithDeviceLayout.Section>
            </>
        );
    }, [actions, partnerReservationNumbers, partnerReservationNumbersQa]);

    if (!segments) {
        return null;
    }

    return (
        <TrainsOrderSegments
            className={className}
            segments={segments}
            tripInfo={tripInfo}
            footerNode={footerNode}
            canVisibleCoachInfo
            showETicketBadge
            canVisibleNonRefundableLabel
            {...prepareQaAttributes(props)}
        />
    );
};

export default Segments;
