import React from 'react';
import {useSelector} from 'react-redux';

import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import totalPriceSelector from 'projects/account/pages/Order/components/GenericOrderTrains/components/TotalPrice/selectors/totalPriceSelector';

import {ORDER_STATUS_VALUES} from 'projects/account/lib/orders/statuses';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/account-OrderTrains-Price';

import Price from 'components/Price/Price';
import TotalPriceReceipts from '../../../GenericOrder/components/TotalPriceReceipts/TotalPriceReceipts';

import cx from './TotalPrice.scss';

interface ITotalPriceProps extends IWithQaAttributes {}

const TotalPrice: React.FC<ITotalPriceProps> = props => {
    const deviceType = useDeviceType();
    const {
        status,
        hasInsurance,
        orderTotalPrice,
        orderTotalRefundAmount,
        paymentReceipt,
        refundReceipts,
    } = useSelector(totalPriceSelector);

    const handlePaymentReceiptDownloadClick = useReachGoal(
        ETrainsGoal.ORDER_DOWNLOAD_PURCHASE_RECEIPT_CLICK,
    );

    if (!orderTotalPrice) {
        return null;
    }

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <div className={cx('price')}>
                {insertJSXIntoKey(i18nBlock.price)({
                    hasInsurance,
                    isDesktop: deviceType.isDesktop,
                    price: (
                        <Price
                            className={cx('value')}
                            {...orderTotalPrice}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'total',
                            })}
                        />
                    ),
                })}
            </div>
            {orderTotalRefundAmount?.value && (
                <div className={cx('refund')}>
                    {insertJSXIntoKey(i18nBlock.refund)({
                        isDesktop: deviceType.isDesktop,
                        price: (
                            <Price
                                {...orderTotalRefundAmount}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'refund',
                                })}
                            />
                        ),
                    })}
                </div>
            )}
            {status === ORDER_STATUS_VALUES.DONE && (
                <TotalPriceReceipts
                    hasInsurance={hasInsurance}
                    paymentReceipt={paymentReceipt}
                    refundReceipts={refundReceipts}
                    onPaymentReceiptClick={handlePaymentReceiptDownloadClick}
                    {...prepareQaAttributes(props)}
                />
            )}
        </div>
    );
};

export default TotalPrice;
