import React, {useEffect, useState} from 'react';
import moment from 'moment';

import {MINUTE} from 'utilities/dateUtils/constants';

import {getNow} from 'utilities/dateUtils';

import * as i18n from 'i18n/common-time';

interface IMinuteTimerProps {
    /**
     * Время до события
     */
    timeLeftAt: string | undefined;
}

/**
 * Компонент показывает количество минут до события
 * и сам обновляется когда число минут уменьшается
 */
const MinuteTimer: React.FC<IMinuteTimerProps> = props => {
    const {timeLeftAt} = props;

    const [minutesLeft, setMinutesLeft] = useState(0);

    useEffect(() => {
        if (!timeLeftAt) {
            return;
        }

        function updateMinutesLeft(): void {
            const minutesCount = moment(timeLeftAt).diff(
                moment(getNow()),
                'minutes',
            );

            setMinutesLeft(Math.max(1, minutesCount));
        }

        updateMinutesLeft();

        const intervalID = setInterval(updateMinutesLeft, MINUTE);

        return (): void => {
            clearInterval(intervalID);
        };
    }, [timeLeftAt]);

    if (!timeLeftAt) {
        return null;
    }

    return (
        <>
            {i18n.minutesDashPeriod({
                count: minutesLeft,
            })}
        </>
    );
};

export default MinuteTimer;
