import {createSelector} from 'reselect';
import uniqBy from 'lodash/uniqBy';

import {isNotUndefined} from 'types/utilities';
import ETrainWarningMessageCode, {
    isTrainWarningMessageCode,
} from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainWarningInfo/ETrainWarningMessageCode';
import EWarningItemParam from '../types/EWarningItemParam';

import trainServicesSelector from 'selectors/account/order/trains/trainServicesSelector';

import getParametersByCode from 'projects/account/pages/Order/components/GenericOrderTrains/components/Warnings/utilities/getParametersByCode';

export interface IWarningItem {
    /**
     * Идентификатор предупреждения, чтобы не показывать одинаковые предупреждения
     */
    key: string;

    [EWarningItemParam.WARNING_CODE]: ETrainWarningMessageCode;
    [EWarningItemParam.WARNING_TO]: string;
    /**
     * Станция отправления пассажира
     */
    [EWarningItemParam.DEPARTURE_STATION]: string;
    /**
     * Станция отправления поезда
     */
    [EWarningItemParam.START_STATION]: string;
    [EWarningItemParam.TRAIN_NUMBER]: string;
}

const orderWarningsSelector = createSelector(
    [trainServicesSelector],
    (services): IWarningItem[] => {
        const warningItems = services
            .map(service => {
                const {trainInfo} = service;
                const {
                    warnings,
                    stationFrom,
                    trainInfo: {
                        trainTicketNumber,
                        startStationTitle,
                        startSettlementTitle,
                    },
                } = trainInfo;

                const warning = warnings?.[0];

                if (!warning || !isTrainWarningMessageCode(warning.code)) {
                    return undefined;
                }

                const warningItem = {
                    warningCode: warning.code,
                    warningTo: warning.to,
                    departureStation:
                        stationFrom.popularTitle ||
                        stationFrom.title ||
                        stationFrom.settlementTitle,
                    startStation: startStationTitle || startSettlementTitle,
                    trainNumber: trainTicketNumber,
                };

                const key = getParametersByCode(warningItem.warningCode)
                    .map(param => warningItem[param])
                    .join();

                return {
                    key,
                    ...warningItem,
                };
            })
            .filter(isNotUndefined);

        /**
         * Не показываем одинаковые предупреждения
         *
         */
        return uniqBy(warningItems, item => item.key);
    },
);

export default orderWarningsSelector;
