import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/account-OrderContacts';

import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Separator from 'components/Separator/Separator';
import Heading from 'components/Heading/Heading';
import Link from 'components/Link/Link';

import cx from './OrderContacts.scss';

interface IOrderTrainsContactsProps extends IWithClassName, IWithQaAttributes {
    email: string;
    phone: string;
    info?: React.ReactNode;
}

const OrderContacts: React.FC<IOrderTrainsContactsProps> = props => {
    const {className, email, phone, info = null} = props;

    const deviceType = useDeviceType();

    return (
        <CardWithDeviceLayout
            className={cx('wrap', deviceMods('wrap', deviceType), className)}
            sectioned
            {...prepareQaAttributes(props)}
        >
            <CardWithDeviceLayout.Section className={cx('top')}>
                <Heading level={deviceType.isMobile ? 3 : 2}>
                    {i18nBlock.title()}
                </Heading>
                {info && (
                    <div
                        className={cx('info')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'info',
                        })}
                    >
                        {info}
                    </div>
                )}
            </CardWithDeviceLayout.Section>
            <Separator />
            <CardWithDeviceLayout.Section className={cx('contacts')}>
                <div className={cx('item')}>
                    <div className={cx('label')}>{i18nBlock.email()}</div>
                    <div className={cx('value')}>
                        <Link
                            url={`mailto:${email}`}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'email',
                            })}
                        >
                            {email}
                        </Link>
                    </div>
                </div>
                <div className={cx('item')}>
                    <div className={cx('label')}>{i18nBlock.phone()}</div>
                    <div
                        className={cx('value')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'phone',
                        })}
                    >
                        {phone}
                    </div>
                </div>
            </CardWithDeviceLayout.Section>
        </CardWithDeviceLayout>
    );
};

export default React.memo(OrderContacts);
