import React, {useCallback, useMemo} from 'react';
import {RouteComponentProps} from 'react-router-dom';
import _flow from 'lodash/flow';

import {
    CANCELLED_WITH_REFUND,
    FAILED,
    IN_PROGRESS,
    INITIAL,
    REFUNDED,
} from 'projects/hotels/constants/hotelsBookingStatuses';

import {IWithDeviceType} from 'src/types/withDeviceType';
import {IOrder} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';

import getOfferInfoByToken from 'sagas/hotels/bookAndPay/getOfferInfoByToken';
import pollingOrder from 'sagas/hotels/bookAndPay/pollingOrder';
import cancelOrder from 'sagas/hotels/bookAndPay/cancelOrder';

import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-Order';

import withSaga from 'containers/withSaga/withSaga';

import BookLoader from 'components/BookLoader/BookLoader';
import DesktopSupportPhone from 'projects/trips/components/DesktopSupportPhone/DesktopSupportPhone';
import Flex from 'components/Flex/Flex';
import OrderBackLink from 'projects/account/pages/Order/components/OrderBackLink/OrderBackLink';
import OrderHotelsDeferredPayment from './components/OrderHotelsDeferredPayment/OrderHotelsDeferredPayment';
import OrderHotelsGuests from './components/OrderHotelsGuests/OrderHotelsGuests';
import OrderHotelsInfo from './components/OrderHotelsInfo/OrderHotelsInfo';
import OrderHotelsActions from './components/OrderHotelsActions/OrderHotelsActions';
import OrderHotelsCancelledInfo from './components/OrderHotelsCancelledInfo/OrderHotelsCancelledInfo';
import OrderHotelsError from './components/OrderHotelsError/OrderHotelsError';
import OrderHotelMainInfo from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelMainInfo/OrderHotelMainInfo';

import bookInfoProvider from 'projects/hotels/containers/BookInfoProvider/BookInfoProvider';

import cx from './OrderHotels.scss';

interface IOrderHotelsProps extends RouteComponentProps, IWithDeviceType {
    orderId: string;
    orderInfo: IOrder;
    offerInfoByToken: {
        offerInfo: IBookOffer;
    };
    startOrderPolling: (data: {orderId: string}) => void;
}

const ROOT_QA = 'hotelOrder';

const OrderHotels: React.FC<IOrderHotelsProps> = props => {
    const {
        orderId: orderIdFromProps,
        deviceType,
        orderInfo,
        orderInfo: {
            orderId: orderIdFromOrder,
            status,
            guestsInfo,
            confirmationInfo,
            payment,
            orderPriceInfo,
            tripId,
        },
        offerInfoByToken: {offerInfo},
        startOrderPolling,
    } = props;

    const inactive = useMemo(() => {
        return [FAILED, REFUNDED, CANCELLED_WITH_REFUND].includes(status);
    }, [status]);

    const handleStartOrderPolling = useCallback((): void => {
        startOrderPolling({orderId: orderIdFromProps});
    }, [orderIdFromProps, startOrderPolling]);

    const renderDeferredPayment = useCallback(() => {
        if (!orderIdFromOrder || !payment?.next || !orderPriceInfo?.price) {
            return null;
        }

        return (
            <OrderHotelsDeferredPayment
                className={cx('deferredPayment')}
                payment={payment}
                orderId={orderIdFromOrder}
                totalPrice={orderPriceInfo.price}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'deferredPayment',
                })}
            />
        );
    }, [orderIdFromOrder, orderPriceInfo, payment]);

    if (
        status === INITIAL ||
        status === IN_PROGRESS ||
        orderIdFromProps !== orderIdFromOrder
    ) {
        return (
            <BookLoader
                title={i18nBlock.loaderDotTitle()}
                description={i18nBlock.loaderDotDescription()}
                isLoading={true}
            />
        );
    }

    const actions = (
        <OrderHotelsActions
            className={cx('actions')}
            orderId={orderIdFromOrder}
            yandexOrderId={confirmationInfo.yandexOrderId}
            status={status}
            isCancelEnabled={offerInfo.cancellationInfo.refundable}
            startOrderPolling={handleStartOrderPolling}
            inactive={inactive}
            {...prepareQaAttributes({
                parent: ROOT_QA,
                current: 'hotelActions',
            })}
        />
    );

    const {hotelInfo} = offerInfo;

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), {
                root_inactive: inactive,
            })}
            flexDirection="column"
            {...prepareQaAttributes(ROOT_QA)}
        >
            <div className={cx('mainContent')}>
                <Flex
                    between={6}
                    inline
                    flexWrap="nowrap"
                    alignItems="baseline"
                    justifyContent="space-between"
                >
                    <OrderBackLink tripId={tripId} />

                    {deviceType.isDesktop && (
                        <DesktopSupportPhone
                            className={cx('supportPhone')}
                            {...prepareQaAttributes({
                                parent: ROOT_QA,
                                current: 'desktopSupportPhone',
                            })}
                        />
                    )}
                </Flex>
                {renderDeferredPayment()}
                <OrderHotelsCancelledInfo
                    className={cx('cancelledInfo')}
                    offerInfo={offerInfo}
                    orderInfo={orderInfo}
                />
                <OrderHotelsError
                    className={cx('error')}
                    status={status}
                    yandexOrderId={confirmationInfo.yandexOrderId}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'error',
                    })}
                />
                <OrderHotelMainInfo
                    className={cx('mainInfo')}
                    orderInfo={orderInfo}
                    offerInfo={offerInfo}
                    actions={deviceType.isMobile ? actions : null}
                    inactive={inactive}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'mainInfo',
                    })}
                />
                {deviceType.isDesktop && actions}
            </div>
            <OrderHotelsInfo
                className={cx('hotelInfo')}
                settlementInfo={hotelInfo.settlementInfo}
                roomInfo={offerInfo.roomInfo}
                bedsGroups={offerInfo.bedsGroups}
                mealInfo={offerInfo.mealInfo}
                inactive={inactive}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'hotelInfo',
                })}
            />
            <OrderHotelsGuests
                className={cx('guests')}
                guests={guestsInfo.guests}
                inactive={inactive}
                {...prepareQaAttributes({parent: ROOT_QA, current: 'guests'})}
            />
        </Flex>
    );
};

export default _flow(
    React.memo,
    withSaga([getOfferInfoByToken, pollingOrder, cancelOrder]),
    bookInfoProvider({
        canFetchOfferInfo: false,
        canStartOrderPolling: true,
    }),
)(OrderHotels);
