import React, {useMemo} from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';
import Modal, {EModalAnimationType} from 'components/Modal/Modal';
import Text from 'components/Text/Text';

import cx from './DetailsModal.scss';

interface IDetailsModalProps extends IWithQaAttributes {
    title: string;
    backButtonText: string | undefined;
    children: React.ReactNode;
    isVisible: boolean;
    onClose(): void;
}

const DetailsModal: React.FC<IDetailsModalProps> = props => {
    const {title, backButtonText, children, isVisible, onClose} = props;

    const rootQa = getQa(props);

    const deviceType = useDeviceType();

    const content = useMemo(() => {
        if (!children) {
            return null;
        }

        return (
            <Modal.Content
                className={cx('modalContent')}
                {...(deviceType.isMobile
                    ? {
                          inset: 0,
                          x: 4,
                          y: 1,
                      }
                    : {})}
            >
                <Text size="xl" weight="bold">
                    {title}
                </Text>
                <div className={cx('details')}>{children}</div>
            </Modal.Content>
        );
    }, [children, deviceType.isMobile, title]);

    const rootCx = cx('root', deviceMods('root', deviceType));

    if (deviceType.isMobile) {
        return (
            <ModalWithBackButton
                className={rootCx}
                isVisible={isVisible}
                onClose={onClose}
                backButtonText={backButtonText}
                fullScreen
                animation={EModalAnimationType.PUSH_RIGHT}
                canRenderBackButton
                verticalAlign="top"
                {...prepareQaAttributes(rootQa)}
            >
                {content}
            </ModalWithBackButton>
        );
    }

    return (
        <Modal
            className={rootCx}
            isVisible={isVisible}
            onClose={onClose}
            {...prepareQaAttributes(rootQa)}
        >
            {content}
        </Modal>
    );
};

export default React.memo(DetailsModal);
