import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IOrder} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {ICoordinates} from 'types/common/ICoordinates';

import getOrderStatusByDisplayState from 'components/OrderHeader/utilities/getOrderStatusByDisplayState';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';

import OrderStatus from 'projects/account/components/OrderStatus/OrderStatus';
import OrderHotelName from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelName/OrderHotelName';
import HotelAddress from 'projects/hotels/components/HotelAddress/HotelAddress';
import CheckinCheckoutDates from 'projects/account/pages/Order/components/OrderHotels/components/CheckinCheckoutDates/CheckinCheckoutDates';
import OrderHotelsRefundInfo from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsRefundInfo/OrderHotelsRefundInfo';
import OrderHotelsEmail from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsEmail/OrderHotelsEmail';
import OrderHotelsOrderId from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelsOrderId/OrderHotelsOrderId';
import Card from 'components/Card/Card';
import HotelMapModal from 'projects/account/components/HotelMapModal/HotelMapModal';
import Separator from 'components/Separator/Separator';
import OrderHotelsPrice from 'projects/account/pages/Order/components/OrderHotels/components/OrderHotelMainInfo/components/OrderHotelsPrice/OrderHotelsPrice';

import cx from './OrderHotelMainInfo.scss';

interface IOrderHotelMainInfoProps extends IWithClassName, IWithQaAttributes {
    orderInfo: IOrder;
    offerInfo: IBookOffer;
    actions: React.ReactNode;
    inactive: boolean;
}

const OrderHotelMainInfo: React.FC<IOrderHotelMainInfoProps> = props => {
    const {className, actions, orderInfo, offerInfo, inactive} = props;

    const {displayState, guestsInfo, confirmationInfo, refundableInfo} =
        orderInfo;
    const {searchParams, hotelInfo} = offerInfo;

    const deviceType = useDeviceType();

    const {
        value: mapIsOpened,
        setFalse: closeMap,
        setTrue: openMap,
    } = useBoolean(false);

    const coordinates = useMemo((): ICoordinates | undefined => {
        const _coordinates = offerInfo?.hotelInfo?.coordinates;

        if (!_coordinates) {
            return;
        }

        const {lat, lng} = _coordinates;

        return {lat, lon: lng};
    }, [offerInfo]);

    return (
        <Card
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_inactive: inactive,
            })}
            shadow={deviceType.isDesktop ? 'default' : undefined}
            inset={deviceType.isDesktop ? 8 : undefined}
            {...prepareQaAttributes(props)}
        >
            <OrderStatus
                className={cx('status')}
                status={getOrderStatusByDisplayState(displayState)}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'status',
                })}
            />
            <OrderHotelName
                className={cx('hotelName')}
                name={hotelInfo.name}
                stars={hotelInfo.stars}
                hotelSlug={hotelInfo.hotelSlug}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'hotelName',
                })}
            />
            <HotelAddress
                className={cx('hotelAddress')}
                onClick={openMap}
                type="link"
                text={hotelInfo.address}
                multiLine
                {...prepareQaAttributes({
                    parent: props,
                    current: 'hotelAddress',
                })}
            />
            <CheckinCheckoutDates
                className={cx('checkinCheckoutDates')}
                checkin={hotelInfo.checkin}
                checkout={hotelInfo.checkout}
                checkinDate={searchParams.startDate}
                checkoutDate={searchParams.endDate}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'checkinCheckoutDates',
                })}
            />
            <Separator className={cx('priceSeparator')} />
            {!inactive && (
                <OrderHotelsPrice
                    className={cx('price')}
                    orderInfo={orderInfo}
                    offerInfo={offerInfo}
                    inactive={inactive}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'orderHotelsPrice',
                    })}
                />
            )}
            {!inactive && (
                <OrderHotelsRefundInfo
                    className={cx('refundInfo')}
                    refundableInfo={refundableInfo}
                    cancellationInfoAll={offerInfo.cancellationInfoAll}
                />
            )}
            {actions && <div className={cx('actions')}>{actions}</div>}
            <OrderHotelsEmail
                className={cx('email')}
                email={guestsInfo.customerEmail}
            />
            <OrderHotelsOrderId
                className={cx('orderId')}
                prettyOrderId={confirmationInfo.yandexOrderId}
            />
            {coordinates && (
                <HotelMapModal
                    visible={mapIsOpened}
                    title={hotelInfo.name}
                    coordinates={coordinates}
                    onClose={closeMap}
                />
            )}
        </Card>
    );
};

export default OrderHotelMainInfo;
