import React from 'react';
import {useSelector} from 'react-redux';

import {IOrder} from 'server/api/HotelsBookAPI/types/IOrder';
import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'types/withClassName';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';

import amountPaidAndTotalPriceAfterPlusWithdrawSelector from 'selectors/hotels/book/amountPaidAndTotalPriceAfterPlusWithdrawSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import TotalPrice from './components/TotalPrice/TotalPrice';
import PlusInfo from './components/PlusInfo/PlusInfo';
import ReceiptsAndDocs from './components/ReceiptsAndDocs/ReceiptsAndDocs';

interface IOrderHotelsPrice extends IWithClassName, IWithQaAttributes {
    orderInfo: IOrder;
    offerInfo: IBookOffer;
    inactive: boolean;
}

const OrderHotelsPrice: React.FC<IOrderHotelsPrice> = props => {
    const {className, orderInfo, offerInfo, inactive} = props;
    const {legalInfo, priceInfo} = offerInfo;
    const {orderPriceInfo, payment, appliedPromoCampaigns} = orderInfo;
    const {receipts} = payment;

    const rootQa = getQa(props);

    const deviceType = useDeviceType();

    const {amountPaid, totalPrice} = useSelector(
        amountPaidAndTotalPriceAfterPlusWithdrawSelector,
    );

    if (!totalPrice) {
        return null;
    }

    return (
        <Flex
            className={className}
            inline
            justifyContent="space-between"
            alignItems="baseline"
            {...prepareQaAttributes(rootQa)}
        >
            <Flex
                between={deviceType.isDesktop ? 8 : 4}
                inline={deviceType.isDesktop}
                flexDirection={deviceType.isDesktop ? 'row' : 'column'}
            >
                <TotalPrice
                    amountPaid={amountPaid}
                    totalPrice={totalPrice}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'totalPrice',
                    })}
                />
                {!inactive && (
                    <PlusInfo
                        direction="column"
                        between={1}
                        justifyContent="flex-start"
                        appliedPromoCampaigns={appliedPromoCampaigns}
                        yandexPlusMode={EYandexPlusApplicationMode.TOPUP}
                    />
                )}
            </Flex>
            <ReceiptsAndDocs
                orderInfo={orderInfo}
                offerInfo={offerInfo}
                priceInfo={priceInfo}
                receipts={receipts}
                orderPriceInfo={orderPriceInfo}
                legalInfo={legalInfo}
                appliedPromoCampaigns={appliedPromoCampaigns}
                inactive={inactive}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'receiptsAndDocs',
                })}
            />
        </Flex>
    );
};

export default React.memo(OrderHotelsPrice);
