import React, {useMemo} from 'react';

import {PAY_FISCAL_RECEIPT_TYPE} from 'projects/hotels/constants/EFiscalReceiptType';

import {
    IBookOffer,
    IBookOfferPriceInfo,
    TBookLegalInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {
    IOrder,
    IOrderPaymentReceipt,
    IOrderPriceInfo,
} from 'server/api/HotelsBookAPI/types/IOrder';
import IActionItem from 'components/ActionsDialog/types/IActionItem';
import {IAppliedPromoCampaigns} from 'types/hotels/offer/IHotelOffer';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import LinkButton from 'components/LinkButton/LinkButton';
import ActionsDialog from 'components/ActionsDialog/ActionsDialog';
import DetailsModal from './components/DetailsModal/DetailsModal';
import FiscalReceiptModal from 'projects/hotels/components/FiscalReceipt/components/FiscalReceiptModal/FiscalReceiptModal';
import useGenerateBusinessTripPdfActions from 'projects/account/pages/Order/components/OrderHotels/hooks/useGenerateBusinessTripPdfActions';
import useDetailsActions from 'projects/account/pages/Order/components/OrderHotels/hooks/useDetailsActions';
import useFiscalReceiptsActions from 'projects/account/pages/Order/components/OrderHotels/hooks/useFiscalReceiptsActions/useFiscalReceiptsActions';
import useActionsDialog from 'projects/account/pages/Order/components/OrderHotels/hooks/useActionsDialog';

import cx from './ReceiptsAndDocs.scss';

interface IReceiptsAndDocsProps extends IWithQaAttributes {
    orderInfo: IOrder;
    offerInfo: IBookOffer;
    priceInfo: IBookOfferPriceInfo;
    receipts: IOrderPaymentReceipt[] | undefined;
    orderPriceInfo: IOrderPriceInfo | null;
    legalInfo: TBookLegalInfo;
    appliedPromoCampaigns: IAppliedPromoCampaigns | undefined;
    inactive: boolean;
}

const ReceiptsAndDocs: React.FC<IReceiptsAndDocsProps> = props => {
    const {
        orderInfo,
        offerInfo,
        priceInfo,
        receipts,
        orderPriceInfo,
        legalInfo,
        appliedPromoCampaigns,
        inactive,
    } = props;
    const {orderId, canGenerateBusinessTripDoc} = orderInfo;

    const actionsDialog = useActionsDialog();

    const orderDetailsActions = useDetailsActions({
        title: i18nBlock.orderDetails(),
    });

    const fiscalReceiptsActions = useFiscalReceiptsActions({
        receipts,
        types: PAY_FISCAL_RECEIPT_TYPE,
        showPlaceholder: false,
    });

    const generateBusinessTripPdfActions = useGenerateBusinessTripPdfActions({
        orderId,
        canGenerateBusinessTripDoc,
        inactive,
    });

    const actions = useMemo((): IActionItem[] => {
        return [
            ...orderDetailsActions.actions,
            ...fiscalReceiptsActions.actions,
            ...generateBusinessTripPdfActions.actions,
        ];
    }, [
        generateBusinessTripPdfActions.actions,
        orderDetailsActions.actions,
        fiscalReceiptsActions.actions,
    ]);

    return (
        <div className={cx('root')} {...prepareQaAttributes(props)}>
            <LinkButton
                innerRef={actionsDialog.ref}
                onClick={actionsDialog.open}
                {...prepareQaAttributes({parent: props, current: 'button'})}
            >
                {i18nBlock.receiptsAndDocs()}
            </LinkButton>
            <ActionsDialog
                {...actionsDialog.commonProps}
                actions={actions}
                isOpened={actionsDialog.isOpened}
                onClose={actionsDialog.close}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'actionsDialog',
                })}
            />
            <DetailsModal
                offerInfo={offerInfo}
                priceInfo={priceInfo}
                orderPriceInfo={orderPriceInfo}
                legalInfo={legalInfo}
                isVisible={orderDetailsActions.isVisible}
                appliedPromoCampaigns={appliedPromoCampaigns}
                onClose={orderDetailsActions.close}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'detailsModal',
                })}
            />
            <FiscalReceiptModal
                backButtonText={offerInfo?.hotelInfo?.name}
                fiscalReceiptUrl={fiscalReceiptsActions.visibleReceiptsUrl}
                isVisible={fiscalReceiptsActions.isVisible}
                onClose={fiscalReceiptsActions.close}
            />
        </div>
    );
};

export default ReceiptsAndDocs;
