import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {
    IBookOffer,
    IBookOfferPriceInfo,
    TBookLegalInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IOrderPriceInfo} from 'server/api/HotelsBookAPI/types/IOrder';
import {IAppliedPromoCampaigns} from 'types/hotels/offer/IHotelOffer';

import amountPaidAndTotalPriceAfterPlusWithdrawSelector from 'selectors/hotels/book/amountPaidAndTotalPriceAfterPlusWithdrawSelector';

import getTaxesAndFeesSum from 'projects/account/pages/Order/components/OrderHotels/utilities/getTaxesAndFeesSum';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import CommonDetailsModal from 'projects/account/pages/Order/components/OrderHotels/components/DetailsModal/DetailsModal';
import Details from './components/Details/Details';

interface IDetailsModalProps extends IWithQaAttributes {
    offerInfo: IBookOffer;
    priceInfo: IBookOfferPriceInfo;
    orderPriceInfo: IOrderPriceInfo | null;
    legalInfo: TBookLegalInfo;
    isVisible: boolean;
    appliedPromoCampaigns: IAppliedPromoCampaigns | undefined;
    onClose(): void;
}

const DetailsModal: React.FC<IDetailsModalProps> = props => {
    const {
        offerInfo,
        priceInfo,
        orderPriceInfo,
        legalInfo,
        isVisible,
        appliedPromoCampaigns,
        onClose,
    } = props;

    const rootQa = getQa(props);

    const {
        hotelInfo: {name: hotelName},
    } = offerInfo;

    const {amountPaid, totalPrice} = useSelector(
        amountPaidAndTotalPriceAfterPlusWithdrawSelector,
    );

    const content = useMemo(() => {
        const hotelCharges = priceInfo?.hotelCharges;
        const hotelPrice = hotelCharges?.totals?.hotelBase ?? '';
        const nights = hotelCharges?.nightly ?? [];
        const discount = hotelCharges.totals?.discount;
        const codeApplicationResults =
            orderPriceInfo?.codeApplicationResults || [];
        const taxesAndFeesSum = getTaxesAndFeesSum(priceInfo);

        if (!hotelPrice || !totalPrice) {
            return null;
        }

        return (
            <Details
                amountPaid={amountPaid}
                totalPrice={totalPrice}
                hotelPrice={hotelPrice}
                nights={nights}
                discount={discount}
                taxesAndFeesSum={taxesAndFeesSum}
                codeApplicationResults={codeApplicationResults}
                legalInfo={legalInfo}
                appliedPromoCampaigns={appliedPromoCampaigns}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'details',
                })}
            />
        );
    }, [
        amountPaid,
        appliedPromoCampaigns,
        legalInfo,
        orderPriceInfo?.codeApplicationResults,
        priceInfo,
        rootQa,
        totalPrice,
    ]);

    return (
        <CommonDetailsModal
            title={i18nBlock.orderDetails()}
            backButtonText={hotelName}
            isVisible={isVisible}
            onClose={onClose}
        >
            {content}
        </CommonDetailsModal>
    );
};

export default React.memo(DetailsModal);
